'use client';

import { useState, useEffect } from 'react';
import { Modal } from '@/components/common/ui';
import PermissionSelector from '@/components/admin/forms/PermissionSelector';
import { UserPagePermissions } from '@/lib/admin/page-permissions';
import { roleLabels } from '@/lib/admin/permissions';

interface EditPermissionsModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSave: (role: 'admin' | 'editor' | 'viewer', permissions: UserPagePermissions | null) => void;
  userName: string;
  userEmail: string;
  currentRole: 'admin' | 'editor' | 'viewer';
  currentPermissions: UserPagePermissions | null;
  isPending?: boolean;
}

export default function EditPermissionsModal({
  isOpen,
  onClose,
  onSave,
  userName,
  userEmail,
  currentRole,
  currentPermissions,
  isPending = false,
}: EditPermissionsModalProps) {
  const [selectedRole, setSelectedRole] = useState<'admin' | 'editor' | 'viewer'>(currentRole);
  const [customPermissions, setCustomPermissions] = useState<UserPagePermissions | null>(currentPermissions);
  const [useCustomPermissions, setUseCustomPermissions] = useState<boolean>(!!currentPermissions);

  // Reset state when modal opens with new data
  useEffect(() => {
    if (isOpen) {
      setSelectedRole(currentRole);
      setCustomPermissions(currentPermissions);
      setUseCustomPermissions(!!currentPermissions);
    }
  }, [isOpen, currentRole, currentPermissions]);

  const handleSave = () => {
    const permissionsToSave = useCustomPermissions ? customPermissions : null;
    onSave(selectedRole, permissionsToSave);
  };

  const handleRoleChange = (newRole: 'admin' | 'editor' | 'viewer') => {
    setSelectedRole(newRole);
    // If changing to admin, disable custom permissions
    if (newRole === 'admin') {
      setUseCustomPermissions(false);
      setCustomPermissions(null);
    }
  };

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Edit User Permissions"
      size='lg'
    >
      <div className="space-y-6">
        {/* User Info */}
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
          <div className="space-y-1">
            <p className="text-sm font-medium text-gray-700">User</p>
            <p className="text-lg font-semibold text-gray-900">{userName}</p>
            <p className="text-sm text-gray-600">{userEmail}</p>
          </div>
        </div>

        {/* Role Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">
            Role
          </label>
          <div className="grid grid-cols-1 gap-3">
            {/* Admin Role */}
            <button
              onClick={() => handleRoleChange('admin')}
              className={`p-4 border-2 rounded-lg text-left transition-all ${
                selectedRole === 'admin'
                  ? 'border-red-500 bg-red-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between">
                <div>
                  <div className="font-semibold text-gray-900">
                    {roleLabels.admin}
                  </div>
                  <div className="text-sm text-gray-600 mt-1">
                    Full access to all features including user management
                  </div>
                </div>
                {selectedRole === 'admin' && (
                  <div className="ml-4">
                    <svg className="w-6 h-6 text-red-600" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                )}
              </div>
            </button>

            {/* Editor Role */}
            <button
              onClick={() => handleRoleChange('editor')}
              className={`p-4 border-2 rounded-lg text-left transition-all ${
                selectedRole === 'editor'
                  ? 'border-blue-500 bg-blue-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between">
                <div>
                  <div className="font-semibold text-gray-900">
                    {roleLabels.editor}
                  </div>
                  <div className="text-sm text-gray-600 mt-1">
                    Can manage content (teachers, news, students)
                  </div>
                </div>
                {selectedRole === 'editor' && (
                  <div className="ml-4">
                    <svg className="w-6 h-6 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                )}
              </div>
            </button>

            {/* Viewer Role */}
            <button
              onClick={() => handleRoleChange('viewer')}
              className={`p-4 border-2 rounded-lg text-left transition-all ${
                selectedRole === 'viewer'
                  ? 'border-green-500 bg-green-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
            >
              <div className="flex items-center justify-between">
                <div>
                  <div className="font-semibold text-gray-900">
                    {roleLabels.viewer}
                  </div>
                  <div className="text-sm text-gray-600 mt-1">
                    Read-only access to view content
                  </div>
                </div>
                {selectedRole === 'viewer' && (
                  <div className="ml-4">
                    <svg className="w-6 h-6 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                )}
              </div>
            </button>
          </div>
        </div>

        {/* Custom Permissions Toggle (only for non-admin) */}
        {selectedRole !== 'admin' && (
          <div className="border-t pt-4">
            <label className="flex items-center gap-3 cursor-pointer">
              <input
                type="checkbox"
                checked={useCustomPermissions}
                onChange={(e) => {
                  setUseCustomPermissions(e.target.checked);
                  if (!e.target.checked) {
                    setCustomPermissions(null);
                  }
                }}
                className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
              />
              <div>
                <span className="text-sm font-medium text-gray-900">
                  Customize Permissions
                </span>
                <p className="text-xs text-gray-600">
                  Override default role permissions with specific page access
                </p>
              </div>
            </label>
          </div>
        )}

        {/* Permission Selector */}
        {useCustomPermissions && selectedRole !== 'admin' && (
          <div className="border-t pt-4">
            <PermissionSelector
              currentPermissions={customPermissions}
              userRole={selectedRole}
              onPermissionsChange={setCustomPermissions}
            />
          </div>
        )}

        {/* Warning for Admin */}
        {selectedRole === 'admin' && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex gap-3">
              <svg className="w-5 h-5 text-yellow-600 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
              </svg>
              <div>
                <p className="text-sm font-medium text-yellow-900">
                  Administrator Role
                </p>
                <p className="text-sm text-yellow-700 mt-1">
                  Administrators have unrestricted access to all features, including the ability to manage other users. Custom permissions are not applicable.
                </p>
              </div>
            </div>
          </div>
        )}

        {/* Actions */}
        <div className="flex gap-3 justify-end pt-4 border-t">
          <button
            onClick={onClose}
            disabled={isPending}
            className="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors disabled:opacity-50"
          >
            Cancel
          </button>
          <button
            onClick={handleSave}
            disabled={isPending}
            className="px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors disabled:opacity-50"
          >
            {isPending ? 'Saving...' : 'Save Changes'}
          </button>
        </div>
      </div>
    </Modal>
  );
}
