'use client';

import { useState, useEffect } from 'react';
import { useSiteSettings, useUpdateSiteSettings } from '@/hooks/useSiteSettings';

export default function ContactSettingsSection() {
  const { data: settings, isLoading } = useSiteSettings([
    'contact_address_line1',
    'contact_address_line2',
    'contact_phone_primary',
    'contact_phone_whatsapp',
    'contact_email',
    'contact_hours_weekday',
    'contact_hours_saturday',
    'contact_hours_sunday',
    'social_facebook',
    'social_linkedin',
    'social_whatsapp',
    'contact_response_time',
    'contact_best_time',
    'contact_map_embed'
  ]);

  const updateSettings = useUpdateSiteSettings();

  const [formData, setFormData] = useState({
    contact_address_line1: '',
    contact_address_line2: '',
    contact_phone_primary: '',
    contact_phone_whatsapp: '',
    contact_email: '',
    contact_hours_weekday: '',
    contact_hours_saturday: '',
    contact_hours_sunday: '',
    social_facebook: '',
    social_linkedin: '',
    social_whatsapp: '',
    contact_response_time: '',
    contact_best_time: '',
    contact_map_embed: ''
  });

  // Update form when settings load
  useEffect(() => {
    if (settings) {
      setFormData({
        contact_address_line1: settings.contact_address_line1 || '',
        contact_address_line2: settings.contact_address_line2 || '',
        contact_phone_primary: settings.contact_phone_primary || '',
        contact_phone_whatsapp: settings.contact_phone_whatsapp || '',
        contact_email: settings.contact_email || '',
        contact_hours_weekday: settings.contact_hours_weekday || '',
        contact_hours_saturday: settings.contact_hours_saturday || '',
        contact_hours_sunday: settings.contact_hours_sunday || '',
        social_facebook: settings.social_facebook || '',
        social_linkedin: settings.social_linkedin || '',
        social_whatsapp: settings.social_whatsapp || '',
        contact_response_time: settings.contact_response_time || '',
        contact_best_time: settings.contact_best_time || '',
        contact_map_embed: settings.contact_map_embed || ''
      });
    }
  }, [settings]);

  const handleChange = (field: keyof typeof formData, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      await updateSettings.mutateAsync(formData);
      alert('Contact settings saved successfully!');
    } catch (error) {
      console.error('Error saving contact settings:', error);
      alert('Failed to save contact settings');
    }
  };

  if (isLoading) {
    return (
      <div className="bg-white rounded-lg shadow p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-4 bg-gray-200 rounded w-1/4"></div>
          <div className="h-10 bg-gray-200 rounded"></div>
          <div className="h-10 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow">
      <form onSubmit={handleSubmit} className="p-6 space-y-8">
        {/* Contact Information Section */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4 pb-2 border-b">
            Contact Information
          </h2>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Address Line 1
              </label>
              <input
                type="text"
                value={formData.contact_address_line1}
                onChange={(e) => handleChange('contact_address_line1', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Street address, building number"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Address Line 2
              </label>
              <input
                type="text"
                value={formData.contact_address_line2}
                onChange={(e) => handleChange('contact_address_line2', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="City, region, country"
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Primary Phone
                </label>
                <input
                  type="text"
                  value={formData.contact_phone_primary}
                  onChange={(e) => handleChange('contact_phone_primary', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  placeholder="+93123456789"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  WhatsApp Number
                </label>
                <input
                  type="text"
                  value={formData.contact_phone_whatsapp}
                  onChange={(e) => handleChange('contact_phone_whatsapp', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                  placeholder="+93123456789"
                />
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Email Address
              </label>
              <input
                type="email"
                value={formData.contact_email}
                onChange={(e) => handleChange('contact_email', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="school@example.com"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Response Time Message
              </label>
              <input
                type="text"
                value={formData.contact_response_time}
                onChange={(e) => handleChange('contact_response_time', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="We typically respond within 24-48 hours..."
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Best Time to Call
              </label>
              <input
                type="text"
                value={formData.contact_best_time}
                onChange={(e) => handleChange('contact_best_time', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Monday to Friday, 9:00 AM - 3:00 PM"
              />
            </div>
          </div>
        </div>

        {/* Office Hours Section */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4 pb-2 border-b">
            Office Hours
          </h2>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Weekday Hours
              </label>
              <input
                type="text"
                value={formData.contact_hours_weekday}
                onChange={(e) => handleChange('contact_hours_weekday', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Monday - Friday: 8:00 AM - 4:00 PM"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Saturday Hours
              </label>
              <input
                type="text"
                value={formData.contact_hours_saturday}
                onChange={(e) => handleChange('contact_hours_saturday', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Saturday: 9:00 AM - 1:00 PM"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Sunday Hours
              </label>
              <input
                type="text"
                value={formData.contact_hours_sunday}
                onChange={(e) => handleChange('contact_hours_sunday', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="Sunday: Closed"
              />
            </div>
          </div>
        </div>

        {/* Social Media Section */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4 pb-2 border-b">
            Social Media Links
          </h2>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Facebook Page URL
              </label>
              <input
                type="url"
                value={formData.social_facebook}
                onChange={(e) => handleChange('social_facebook', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="https://facebook.com/yourpage"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                LinkedIn Profile URL
              </label>
              <input
                type="url"
                value={formData.social_linkedin}
                onChange={(e) => handleChange('social_linkedin', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="https://linkedin.com/in/yourprofile"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                WhatsApp Contact URL
              </label>
              <input
                type="url"
                value={formData.social_whatsapp}
                onChange={(e) => handleChange('social_whatsapp', e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                placeholder="https://wa.me/123456789"
              />
            </div>
          </div>
        </div>

        {/* Map Section */}
        <div>
          <h2 className="text-xl font-semibold text-gray-900 mb-4 pb-2 border-b">
            Map Settings
          </h2>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-1">
              Google Maps Embed URL
            </label>
            <textarea
              value={formData.contact_map_embed}
              onChange={(e) => handleChange('contact_map_embed', e.target.value)}
              rows={3}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 font-mono text-sm"
              placeholder="https://www.google.com/maps/embed?pb=..."
            />
            <p className="mt-1 text-sm text-gray-500">
              Get this URL from Google Maps → Share → Embed a map
            </p>
          </div>
        </div>

        {/* Save Button */}
        <div className="flex justify-end pt-4 border-t">
          <button
            type="submit"
            disabled={updateSettings.isPending}
            className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
          >
            {updateSettings.isPending ? 'Saving...' : 'Save Contact Settings'}
          </button>
        </div>

        {/* Success/Error Messages */}
        {updateSettings.isSuccess && (
          <div className="mt-4 p-4 bg-green-50 border border-green-200 rounded-lg text-green-800">
            ✅ Contact settings saved successfully!
          </div>
        )}
        {updateSettings.isError && (
          <div className="mt-4 p-4 bg-red-50 border border-red-200 rounded-lg text-red-800">
            ❌ Failed to save contact settings. Please try again.
          </div>
        )}
      </form>
    </div>
  );
}
