'use client';

import { useState, useEffect } from 'react';
import { useCreateHeroSlide, useUpdateHeroSlide, HeroSlide, HeroSlideInput } from '@/hooks/useHeroSlides';
import { InputField, Modal } from '@/components/common/ui';

interface HeroSlideModalProps {
  isOpen: boolean;
  slide?: HeroSlide | null;
  onClose: () => void;
}

export default function HeroSlideModal({ isOpen, slide, onClose }: HeroSlideModalProps) {
  const createMutation = useCreateHeroSlide();
  const updateMutation = useUpdateHeroSlide();
  const [formData, setFormData] = useState<HeroSlideInput>({
    title: '',
    description: '',
    imageUrl: '',
    alt: '',
    order: 0,
    active: true,
  });
  const [error, setError] = useState('');

  useEffect(() => {
    if (slide) {
      setFormData({
        title: slide.title,
        description: slide.description || '',
        imageUrl: slide.imageUrl,
        alt: slide.alt,
        order: slide.order,
        active: slide.active,
      });
    }
  }, [slide]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? (e.target as HTMLInputElement).checked : 
              name === 'order' ? parseInt(value) || 0 : 
              value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError('');

    try {
      if (slide) {
        // Update existing slide
        await updateMutation.mutateAsync({ id: slide.id, ...formData });
      } else {
        // Create new slide
        await createMutation.mutateAsync(formData);
      }
      onClose();
    } catch (err) {
      setError(err instanceof Error ? err.message : 'An error occurred');
    }
  };

  const isLoading = createMutation.isPending || updateMutation.isPending;

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title={slide ? 'Edit Hero Slide' : 'Add Hero Slide'}
      size="lg"
    >
      <form onSubmit={handleSubmit} className="space-y-4">
        {error && (
          <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
            <p className="text-red-800 text-sm">{error}</p>
          </div>
        )}

        {/* Title */}
        <InputField
          id="title"
          name="title"
          label="Title"
          value={formData.title}
          onChange={handleChange}
          required
          placeholder="Enter slide title"
        />

        {/* Description */}
        <div>
          <label htmlFor="description" className="block text-sm font-medium text-gray-700 mb-1">
            Description
          </label>
          <textarea
            id="description"
            name="description"
            value={formData.description}
            onChange={handleChange}
            rows={3}
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Enter slide description"
          />
        </div>

        {/* Image URL */}
        <InputField
          id="imageUrl"
          name="imageUrl"
          label="Image URL"
          value={formData.imageUrl}
          onChange={handleChange}
          required
          placeholder="/images/home/slide-1.jpg"
          helperText="Relative path to image in public folder or full URL"
        />

        {/* Alt Text */}
        <InputField
          id="alt"
          name="alt"
          label="Alt Text"
          value={formData.alt}
          onChange={handleChange}
          required
          placeholder="Descriptive text for accessibility"
          helperText="Describe the image for screen readers"
        />

        {/* Order */}
        <InputField
          id="order"
          name="order"
          type="number"
          label="Display Order"
          value={formData.order}
          onChange={handleChange}
          required
          placeholder="0"
          helperText="Lower numbers appear first"
        />

        {/* Active Checkbox */}
        <div className="flex items-center">
          <input
            type="checkbox"
            id="active"
            name="active"
            checked={formData.active}
            onChange={handleChange}
            className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
          />
          <label htmlFor="active" className="ml-2 block text-sm text-gray-700">
            Active (shown on website)
          </label>
        </div>

        {/* Image Preview */}
        {formData.imageUrl && (
          <div className="mt-4">
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Preview
            </label>
            <div className="w-full h-48 bg-gray-100 rounded-lg overflow-hidden">
              <img
                src={formData.imageUrl}
                alt={formData.alt || 'Preview'}
                className="w-full h-full object-cover"
                onError={(e) => {
                  (e.target as HTMLImageElement).src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="400" height="300"%3E%3Crect fill="%23ddd" width="400" height="300"/%3E%3Ctext fill="%23999" x="50%25" y="50%25" dominant-baseline="middle" text-anchor="middle"%3EImage not found%3C/text%3E%3C/svg%3E';
                }}
              />
            </div>
          </div>
        )}

        {/* Actions */}
        <div className="flex justify-end gap-3 pt-4 border-t border-gray-200">
          <button
            type="button"
            onClick={onClose}
            disabled={isLoading}
            className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200 transition font-medium disabled:opacity-50"
          >
            Cancel
          </button>
          <button
            type="submit"
            disabled={isLoading}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium disabled:opacity-50"
          >
            {isLoading ? 'Saving...' : slide ? 'Update Slide' : 'Create Slide'}
          </button>
        </div>
      </form>
    </Modal>
  );
}
