'use client';

import { useState } from 'react';
import { useHeroSlides, useCreateHeroSlide, useUpdateHeroSlide, useDeleteHeroSlide, HeroSlideInput, HeroSlide } from '@/hooks/useHeroSlides';
import { Plus, Trash2, Edit2, Image as ImageIcon, MoveUp, MoveDown } from 'lucide-react';
import HeroSlideModal from './HeroSlideModal';

export default function HeroSlidesSection() {
  const { data: slides, isLoading } = useHeroSlides();
  const deleteMutation = useDeleteHeroSlide();
  const [modalOpen, setModalOpen] = useState(false);
  const [editingSlide, setEditingSlide] = useState<HeroSlide | null>(null);

  const handleAdd = () => {
    setEditingSlide(null);
    setModalOpen(true);
  };

  const handleEdit = (slide: HeroSlide) => {
    setEditingSlide(slide);
    setModalOpen(true);
  };

  const handleDelete = async (id: string) => {
    if (confirm('Are you sure you want to delete this hero slide?')) {
      try {
        await deleteMutation.mutateAsync(id);
      } catch (error) {
        alert(error instanceof Error ? error.message : 'Failed to delete slide');
      }
    }
  };

  if (isLoading) {
    return (
      <div className="bg-white rounded-lg shadow p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/3 mb-6"></div>
          <div className="space-y-4">
            {[...Array(3)].map((_, i) => (
              <div key={i} className="h-32 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow p-6">
      {/* Header */}
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-xl font-semibold text-gray-900">Hero Slider Management</h2>
          <p className="mt-1 text-sm text-gray-600">
            Manage the images and content displayed on the home page hero slider
          </p>
        </div>
        <button
          onClick={handleAdd}
          className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium"
        >
          <Plus className="h-5 w-5 mr-2" />
          Add Slide
        </button>
      </div>

      {/* Slides List */}
      {!slides || slides.length === 0 ? (
        <div className="text-center py-12">
          <ImageIcon className="h-12 w-12 text-gray-400 mx-auto mb-4" />
          <p className="text-gray-600 mb-4">No hero slides yet</p>
          <button
            onClick={handleAdd}
            className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium"
          >
            <Plus className="h-5 w-5 mr-2" />
            Add Your First Slide
          </button>
        </div>
      ) : (
        <div className="space-y-4">
          {slides.map((slide, index) => (
            <div
              key={slide.id}
              className="flex items-start gap-4 p-4 border border-gray-200 rounded-lg hover:border-gray-300 transition"
            >
              {/* Image Preview */}
              <div className="flex-shrink-0 w-48 h-32 bg-gray-100 rounded-lg overflow-hidden">
                <img
                  src={slide.imageUrl}
                  alt={slide.alt}
                  className="w-full h-full object-cover"
                />
              </div>

              {/* Content */}
              <div className="flex-grow">
                <div className="flex items-start justify-between mb-2">
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900">{slide.title}</h3>
                    {slide.description && (
                      <p className="text-sm text-gray-600 mt-1 line-clamp-2">{slide.description}</p>
                    )}
                  </div>
                  <div className="flex items-center gap-2 ml-4">
                    <span
                      className={`px-2 py-1 text-xs font-medium rounded-full ${
                        slide.active
                          ? 'bg-green-100 text-green-800'
                          : 'bg-gray-100 text-gray-800'
                      }`}
                    >
                      {slide.active ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                </div>

                <div className="flex items-center gap-2 mt-3">
                  <span className="text-xs text-gray-500">Order: {slide.order}</span>
                </div>
              </div>

              {/* Actions */}
              <div className="flex flex-col gap-2 flex-shrink-0">
                <button
                  onClick={() => handleEdit(slide)}
                  className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition"
                  title="Edit"
                >
                  <Edit2 className="h-5 w-5" />
                </button>
                <button
                  onClick={() => handleDelete(slide.id)}
                  disabled={deleteMutation.isPending}
                  className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition disabled:opacity-50"
                  title="Delete"
                >
                  <Trash2 className="h-5 w-5" />
                </button>
              </div>
            </div>
          ))}
        </div>
      )}

      {/* Modal */}
      <HeroSlideModal
        isOpen={modalOpen}
        slide={editingSlide}
        onClose={() => {
          setModalOpen(false);
          setEditingSlide(null);
        }}
      />
    </div>
  );
}
