'use client';

import { useState, useEffect } from 'react';
import { useSiteSettings, useUpdateSiteSettings } from '@/hooks/useSiteSettings';
import { Upload, Image as ImageIcon, Save } from 'lucide-react';
import { InputField } from '@/components/common/ui';

export default function LogoSettingsSection() {
  const { data: settings, isLoading } = useSiteSettings([
    'logo_url',
    'logo_text',
    'logo_initials',
    'favicon_url',
    'logo_alt',
    'show_logo_text',
  ]);
  
  const updateMutation = useUpdateSiteSettings();
  
  const [formData, setFormData] = useState({
    logo_url: '',
    logo_text: 'Setaragan Zirak',
    logo_initials: 'SZ',
    favicon_url: '',
    logo_alt: 'Setaragan Zirak School Logo',
    show_logo_text: 'true',
  });

  const [message, setMessage] = useState('');
  const [error, setError] = useState('');

  useEffect(() => {
    if (settings) {
      setFormData({
        logo_url: settings.logo_url || '',
        logo_text: settings.logo_text || 'Setaragan Zirak',
        logo_initials: settings.logo_initials || 'SZ',
        favicon_url: settings.favicon_url || '',
        logo_alt: settings.logo_alt || 'Setaragan Zirak School Logo',
        show_logo_text: settings.show_logo_text || 'true',
      });
    }
  }, [settings]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type, checked } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? (checked ? 'true' : 'false') : value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError('');
    setMessage('');

    try {
      await updateMutation.mutateAsync(formData);
      setMessage('Logo settings updated successfully!');
      setTimeout(() => setMessage(''), 3000);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to update settings');
    }
  };

  if (isLoading) {
    return (
      <div className="bg-white rounded-lg shadow p-6 animate-pulse">
        <div className="h-8 bg-gray-200 rounded w-1/3 mb-6"></div>
        <div className="space-y-4">
          {[...Array(4)].map((_, i) => (
            <div key={i} className="h-12 bg-gray-200 rounded"></div>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow">
      {/* Header */}
      <div className="p-6 border-b border-gray-200">
        <h2 className="text-xl font-semibold text-gray-900">Logo & Branding</h2>
        <p className="mt-1 text-sm text-gray-600">
          Manage your school logo and branding that appears across the website
        </p>
      </div>

      {/* Form */}
      <form onSubmit={handleSubmit} className="p-6 space-y-6">
        {/* Messages */}
        {message && (
          <div className="p-4 bg-green-50 border border-green-200 rounded-lg">
            <p className="text-green-800 text-sm">{message}</p>
          </div>
        )}
        {error && (
          <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
            <p className="text-red-800 text-sm">{error}</p>
          </div>
        )}

        {/* Logo URL */}
        <div>
          <InputField
            id="logo_url"
            name="logo_url"
            label="Logo Image URL"
            value={formData.logo_url}
            onChange={handleChange}
            placeholder="/images/logo.png"
            helperText="Path to your school logo image (PNG, SVG, or JPG)"
          />
          {formData.logo_url && (
            <div className="mt-3">
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Preview
              </label>
              <div className="p-4 bg-gray-50 rounded-lg inline-block">
                <img
                  src={formData.logo_url}
                  alt={formData.logo_alt}
                  className="h-16 object-contain"
                  onError={(e) => {
                    (e.target as HTMLImageElement).style.display = 'none';
                  }}
                />
              </div>
            </div>
          )}
        </div>

        {/* Logo Text */}
        <InputField
          id="logo_text"
          name="logo_text"
          label="Logo Text"
          value={formData.logo_text}
          onChange={handleChange}
          required
          placeholder="Setaragan Zirak"
          helperText="Text displayed with or without the logo image"
        />

        {/* Logo Initials */}
        <InputField
          id="logo_initials"
          name="logo_initials"
          label="Logo Initials"
          value={formData.logo_initials}
          onChange={handleChange}
          required
          placeholder="SZ"
          maxLength={3}
          helperText="Shown on mobile or when logo image is not available (2-3 characters)"
        />

        {/* Logo Alt Text */}
        <InputField
          id="logo_alt"
          name="logo_alt"
          label="Logo Alt Text"
          value={formData.logo_alt}
          onChange={handleChange}
          required
          placeholder="Setaragan Zirak School Logo"
          helperText="Accessibility description for screen readers"
        />

        {/* Favicon URL */}
        <div>
          <InputField
            id="favicon_url"
            name="favicon_url"
            label="Favicon URL (Optional)"
            value={formData.favicon_url}
            onChange={handleChange}
            placeholder="/images/favicon.ico"
            helperText="Browser tab icon (16x16 or 32x32 .ico or .png file)"
          />
          {formData.favicon_url && (
            <div className="mt-3">
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Favicon Preview
              </label>
              <div className="p-4 bg-gray-50 rounded-lg inline-block">
                <img
                  src={formData.favicon_url}
                  alt="Favicon"
                  className="w-8 h-8 object-contain"
                  onError={(e) => {
                    (e.target as HTMLImageElement).style.display = 'none';
                  }}
                />
              </div>
            </div>
          )}
        </div>

        {/* Show Logo Text Toggle */}
        <div className="flex items-center">
          <input
            type="checkbox"
            id="show_logo_text"
            name="show_logo_text"
            checked={formData.show_logo_text === 'true'}
            onChange={handleChange}
            className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
          />
          <label htmlFor="show_logo_text" className="ml-2 block text-sm text-gray-700">
            Show logo text alongside image
          </label>
        </div>

        {/* Preview Section */}
        <div className="pt-6 border-t border-gray-200">
          <label className="block text-sm font-medium text-gray-700 mb-3">
            Logo Preview (as it will appear)
          </label>
          <div className="p-6 bg-gray-50 rounded-lg">
            <div className="flex items-center gap-3">
              {formData.logo_url ? (
                <img
                  src={formData.logo_url}
                  alt={formData.logo_alt}
                  className="h-10 object-contain"
                  onError={(e) => {
                    (e.target as HTMLImageElement).style.display = 'none';
                  }}
                />
              ) : (
                <div className="w-10 h-10 bg-blue-600 rounded-lg flex items-center justify-center text-white font-bold text-lg">
                  {formData.logo_initials}
                </div>
              )}
              {formData.show_logo_text === 'true' && (
                <div className="flex items-baseline">
                  <span className="text-xl font-bold text-gray-900">
                    {formData.logo_text.split(' ')[0]}
                  </span>
                  {formData.logo_text.split(' ')[1] && (
                    <span className="text-xl font-bold text-blue-600 ml-1">
                      {formData.logo_text.split(' ').slice(1).join(' ')}
                    </span>
                  )}
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Submit Button */}
        <div className="flex justify-end gap-3 pt-4">
          <button
            type="submit"
            disabled={updateMutation.isPending}
            className="inline-flex items-center gap-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition font-medium disabled:opacity-50 disabled:cursor-not-allowed"
          >
            <Save className="h-5 w-5" />
            {updateMutation.isPending ? 'Saving...' : 'Save Logo Settings'}
          </button>
        </div>
      </form>
    </div>
  );
}
