'use client';

import { ReactNode } from 'react';
import { useSession } from 'next-auth/react';
import { useRouter } from 'next/navigation';
import { useEffect, useState } from 'react';

interface AdminAuthGuardProps {
  children: ReactNode;
  requiredRole?: 'admin' | 'editor' | 'viewer';
}

/**
 * Higher-order component that protects admin pages
 * Ensures user is authenticated and has required role
 * Redirects to login if not authenticated
 * Shows loading state while checking auth
 */
export default function AdminAuthGuard({
  children,
  requiredRole = 'editor', // Default to editor, admin pages can require 'admin'
}: AdminAuthGuardProps) {
  const { data: session, status } = useSession();
  const router = useRouter();
  const [isAuthorized, setIsAuthorized] = useState(false);

  useEffect(() => {
    if (status === 'unauthenticated') {
      // Not logged in - redirect to sign in
      router.replace('/auth/signin');
    } else if (status === 'authenticated') {
      // Check role-based access
      const userRole = session?.user?.role as string;
      const roleHierarchy: Record<string, number> = {
        admin: 3,
        editor: 2,
        viewer: 1,
      };

      const userLevel = roleHierarchy[userRole] || 0;
      const requiredLevel = roleHierarchy[requiredRole] || 0;

      if (userLevel >= requiredLevel) {
        setIsAuthorized(true);
      } else {
        // Insufficient permissions
        router.replace('/auth/error?error=forbidden');
      }
    }
  }, [status, session, router, requiredRole]);

  if (status === 'loading') {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!isAuthorized) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <p className="text-gray-600 mb-4">Checking permissions...</p>
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
        </div>
      </div>
    );
  }

  return <>{children}</>;
}

/**
 * Hook to manually check authentication in admin components
 * Use for conditional rendering of admin features
 */
export function useAdminAuth() {
  const { data: session, status } = useSession();
  const isAdmin = session?.user?.role === 'admin';
  const isEditor = session?.user?.role === 'editor';
  const isAuthenticated = status === 'authenticated';
  const isLoading = status === 'loading';

  return {
    session,
    isAuthenticated,
    isAdmin,
    isEditor,
    isLoading,
    userId: session?.user?.id,
    userRole: session?.user?.role,
  };
}

/**
 * Helper to check if user can perform admin action
 */
export function canPerformAdminAction(userRole?: string): boolean {
  return userRole === 'admin' || userRole === 'editor';
}

/**
 * Helper to check if user is admin-only
 */
export function isAdminOnly(userRole?: string): boolean {
  return userRole === 'admin';
}
