'use client';

import { UserPagePermissions, ADMIN_PAGES_CONFIG, getPermissionSummary } from '@/lib/admin/page-permissions';
import { Popover, PopoverButton, PopoverPanel } from '@headlessui/react';

interface PermissionSummaryProps {
  role: 'admin' | 'editor' | 'viewer';
  permissions: UserPagePermissions | null;
  compact?: boolean;
}

export default function PermissionSummary({ role, permissions, compact = false }: PermissionSummaryProps) {

  // Admin has full access
  if (role === 'admin') {
    return (
      <div className="inline-flex items-center">
        <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-purple-100 text-purple-800 border border-purple-200">
          <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
          </svg>
          Full Access
        </span>
      </div>
    );
  }

  // Count pages with permissions first
  const pageCount = permissions ? Object.keys(permissions).filter(pageId => 
    permissions[pageId] && permissions[pageId].length > 0
  ).length : 0;

  // Get total action count
  const actionCount = permissions ? Object.values(permissions).reduce((total, actions) => 
    total + (actions?.length || 0), 0
  ) : 0;

  // No custom permissions set (null or empty)
  if (!permissions || pageCount === 0) {
    return (
      <div className="inline-flex items-center">
        <span className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 border border-gray-300">
          <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
            <path fillRule="evenodd" d="M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z" clipRule="evenodd" />
          </svg>
          Dashboard Only
        </span>
      </div>
    );
  }

  if (compact) {
    return (
      <Popover>
        <PopoverButton className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-blue-100 text-blue-800 border border-blue-200 hover:bg-blue-200 transition-colors focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-1">
          <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
            <path d="M9 2a1 1 0 000 2h2a1 1 0 100-2H9z" />
            <path fillRule="evenodd" d="M4 5a2 2 0 012-2 3 3 0 003 3h2a3 3 0 003-3 2 2 0 012 2v11a2 2 0 01-2 2H6a2 2 0 01-2-2V5zm3 4a1 1 0 000 2h.01a1 1 0 100-2H7zm3 0a1 1 0 000 2h3a1 1 0 100-2h-3zm-3 4a1 1 0 100 2h.01a1 1 0 100-2H7zm3 0a1 1 0 100 2h3a1 1 0 100-2h-3z" clipRule="evenodd" />
          </svg>
          {pageCount} {pageCount === 1 ? 'page' : 'pages'}
        </PopoverButton>

        <PopoverPanel 
          anchor="bottom start"
          className="z-[9999] bg-white border border-gray-200 rounded-lg shadow-xl p-3 min-w-[280px] max-w-[320px] max-h-[400px] overflow-y-auto mt-1"
        >
          {({ close }) => (
            <>
              <div className="flex items-center justify-between mb-2">
                <span className="text-xs font-semibold text-gray-700">Permission Details</span>
                <button
                  onClick={() => close()}
                  className="text-gray-400 hover:text-gray-600 focus:outline-none"
                >
                  <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                    <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd" />
                  </svg>
                </button>
              </div>
              <div className="space-y-2 text-xs">
                {Object.entries(permissions).map(([pageId, actions]) => {
                  if (!actions || actions.length === 0) return null;
                  const pageConfig = ADMIN_PAGES_CONFIG.find(p => p.id === pageId);
                  if (!pageConfig) return null;

                  return (
                    <div key={pageId} className="border-b border-gray-100 pb-2 last:border-0">
                      <div className="font-medium text-gray-800 mb-1">{pageConfig.name}</div>
                      <div className="flex flex-wrap gap-1">
                        {actions.map(action => (
                          <span
                            key={action}
                            className="inline-flex items-center px-1.5 py-0.5 rounded bg-blue-50 text-blue-700 text-[10px] border border-blue-100"
                          >
                            {action}
                          </span>
                        ))}
                      </div>
                    </div>
                  );
                })}
              </div>
              <div className="mt-2 pt-2 border-t border-gray-200 text-[10px] text-gray-500">
                Total: {actionCount} {actionCount === 1 ? 'permission' : 'permissions'}
              </div>
            </>
          )}
        </PopoverPanel>
      </Popover>
    );
  }

  // Full view (for modal or detailed view)
  const summary = getPermissionSummary(permissions, role);

  return (
    <div className="space-y-2">
      <div className="text-sm font-medium text-gray-700">
        {pageCount} {pageCount === 1 ? 'page' : 'pages'} • {actionCount} {actionCount === 1 ? 'permission' : 'permissions'}
      </div>
      <div className="text-xs text-gray-600 space-y-1">
        {summary.map((item, idx) => (
          <div key={idx}>• {item.pageName}: {item.actions.join(', ')}</div>
        ))}
      </div>
    </div>
  );
}
