"use client";

import { useState } from "react";
import { useApproveUser, useRejectUser, useSuspendUser } from "@/hooks/useAdminUserManagement";

interface UserApprovalModalProps {
  userId: string;
  action: "approve" | "reject" | "suspend";
  onClose: () => void;
}

export function UserApprovalModal({ userId, action, onClose }: UserApprovalModalProps) {
  const [reason, setReason] = useState("");
  const [error, setError] = useState<string | null>(null);

  const approveMutation = useApproveUser();
  const rejectMutation = useRejectUser();
  const suspendMutation = useSuspendUser();

  const isLoading = approveMutation.isPending || rejectMutation.isPending || suspendMutation.isPending;
  const isError = approveMutation.isError || rejectMutation.isError || suspendMutation.isError;

  const handleAction = async () => {
    try {
      setError(null);

      if (action === "approve") {
        await approveMutation.mutateAsync({ userId });
      } else if (action === "reject") {
        await rejectMutation.mutateAsync({ userId, reason });
      } else if (action === "suspend") {
        await suspendMutation.mutateAsync({ userId, reason });
      }

      onClose();
    } catch (err) {
      setError(err instanceof Error ? err.message : "An error occurred");
    }
  };

  const getActionConfig = () => {
    switch (action) {
      case "approve":
        return {
          title: "Approve User",
          description: "This user will be able to login and access the admin panel.",
          buttonColor: "bg-green-600 hover:bg-green-700",
          buttonText: "Approve",
          showReason: false,
        };
      case "reject":
        return {
          title: "Reject User",
          description: "This user will not be able to login. You can provide a reason.",
          buttonColor: "bg-red-600 hover:bg-red-700",
          buttonText: "Reject",
          showReason: true,
        };
      case "suspend":
        return {
          title: "Suspend User",
          description: "This user will be blocked from logging in. You can provide a reason.",
          buttonColor: "bg-orange-600 hover:bg-orange-700",
          buttonText: "Suspend",
          showReason: true,
        };
    }
  };

  const config = getActionConfig();

  return (
    <>
      {/* Overlay */}
      <div
        className="fixed inset-0 bg-black bg-opacity-50 z-40"
        onClick={onClose}
      />

      {/* Modal */}
      <div className="fixed inset-0 z-50 flex items-center justify-center p-4">
        <div className="bg-white rounded-lg shadow-lg max-w-md w-full">
          {/* Header */}
          <div className="px-6 py-4 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">{config.title}</h2>
            <p className="text-sm text-gray-600 mt-1">{config.description}</p>
          </div>

          {/* Body */}
          <div className="px-6 py-4">
            {config.showReason && (
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Reason (optional)
                </label>
                <textarea
                  value={reason}
                  onChange={(e) => setReason(e.target.value)}
                  placeholder="Provide a reason for this action..."
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
            )}

            {error && (
              <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded text-sm text-red-600">
                {error}
              </div>
            )}

            {isError && !error && (
              <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded text-sm text-red-600">
                An error occurred
              </div>
            )}
          </div>

          {/* Footer */}
          <div className="px-6 py-4 border-t border-gray-200 flex gap-3">
            <button
              onClick={onClose}
              disabled={isLoading}
              className="flex-1 px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors disabled:opacity-50"
            >
              Cancel
            </button>
            <button
              onClick={handleAction}
              disabled={isLoading}
              className={`flex-1 px-4 py-2 text-sm font-medium text-white ${config.buttonColor} rounded-lg transition-colors disabled:opacity-50`}
            >
              {isLoading ? "Processing..." : config.buttonText}
            </button>
          </div>
        </div>
      </div>
    </>
  );
}
