'use client';

import React from 'react';
import { useFormik } from 'formik';
import * as Yup from 'yup';
import { useAdmissionsFormSubmission, type AdmissionsFormData } from '@/hooks';

interface AdmissionsFormProps {
  className?: string;
  onSuccess?: () => void;
}

// Validation schema matching original form
const ADMISSIONS_FORM_SCHEMA = Yup.object().shape({
  studentFirstName: Yup.string().required('Student first name is required').min(2).max(100),
  studentLastName: Yup.string().required('Student last name is required').min(2).max(100),
  gradeLevel: Yup.string().required('Grade level is required'),
  birthDate: Yup.string().required('Birth date is required'),
  parentName: Yup.string().required('Parent/Guardian name is required').min(2).max(100),
  parentPhone: Yup.string().required('Phone number is required').matches(/^[\d\s\-\+\(\)]*$/, 'Valid phone required'),
  parentEmail: Yup.string().email('Invalid email').required('Email is required'),
  message: Yup.string().optional().max(5000),
});

const AdmissionsForm: React.FC<AdmissionsFormProps> = ({ className, onSuccess }) => {
  const [statusMessage, setStatusMessage] = React.useState<{
    type: 'success' | 'error' | null;
    message: string;
  }>({
    type: null,
    message: '',
  });

  const initialValues: AdmissionsFormData = {
    studentFirstName: '',
    studentLastName: '',
    gradeLevel: '',
    birthDate: '',
    parentName: '',
    parentPhone: '',
    parentEmail: '',
    message: '',
  };

  // Custom React Query hook for form submission
  const mutation = useAdmissionsFormSubmission(
    (data) => {
      setStatusMessage({
        type: 'success',
        message: data.message || 'Thank you! Your application has been submitted successfully.',
      });

      formik.resetForm();

      if (onSuccess) {
        onSuccess();
      }

      // Clear success message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    },
    (error) => {
      setStatusMessage({
        type: 'error',
        message: error.message || 'An error occurred. Please try again.',
      });

      // Clear error message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    }
  );

  const formik = useFormik({
    initialValues,
    validationSchema: ADMISSIONS_FORM_SCHEMA,
    validateOnChange: true,
    validateOnBlur: true,
    onSubmit: (values) => {
      setStatusMessage({ type: null, message: '' });
      mutation.mutate(values);
    },
  });

  return (
    <form onSubmit={formik.handleSubmit} className={`space-y-6 ${className}`}>
      {/* Status Messages */}
      {statusMessage.type && (
        <div
          className={`p-3 rounded-lg text-sm ${
            statusMessage.type === 'success'
              ? 'bg-green-50 text-green-800 border border-green-200'
              : 'bg-red-50 text-red-800 border border-red-200'
          }`}
        >
          {statusMessage.message}
        </div>
      )}

      {/* Student First & Last Name */}
      <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
        <div>
          <input
            type="text"
            id="studentFirstName"
            {...formik.getFieldProps('studentFirstName')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
            placeholder="Student's First Name"
          />
          {formik.touched.studentFirstName && formik.errors.studentFirstName ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.studentFirstName}</p>
          ) : null}
        </div>
        <div>
          <input
            type="text"
            id="studentLastName"
            {...formik.getFieldProps('studentLastName')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
            placeholder="Student's Last Name"
          />
          {formik.touched.studentLastName && formik.errors.studentLastName ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.studentLastName}</p>
          ) : null}
        </div>
      </div>

      {/* Grade Level & Birth Date */}
      <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
        <div>
          <select
            id="gradeLevel"
            {...formik.getFieldProps('gradeLevel')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
          >
            <option value="" className="text-gray-900">
              Select Grade Level
            </option>
            <option value="1-3" className="text-gray-900">
              Grades 1-3 (Primary)
            </option>
            <option value="4-6" className="text-gray-900">
              Grades 4-6 (Elementary)
            </option>
            <option value="7-9" className="text-gray-900">
              Grades 7-9 (Middle School)
            </option>
            <option value="10-12" className="text-gray-900">
              Grades 10-12 (High School)
            </option>
          </select>
          {formik.touched.gradeLevel && formik.errors.gradeLevel ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.gradeLevel}</p>
          ) : null}
        </div>
        <div>
          <input
            type="date"
            id="birthDate"
            {...formik.getFieldProps('birthDate')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
            placeholder="Student's Birth Date"
          />
          {formik.touched.birthDate && formik.errors.birthDate ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.birthDate}</p>
          ) : null}
        </div>
      </div>

      {/* Parent Name & Phone */}
      <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
        <div>
          <input
            type="text"
            id="parentName"
            {...formik.getFieldProps('parentName')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
            placeholder="Parent/Guardian Name"
          />
          {formik.touched.parentName && formik.errors.parentName ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.parentName}</p>
          ) : null}
        </div>
        <div>
          <input
            type="tel"
            id="parentPhone"
            {...formik.getFieldProps('parentPhone')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
            placeholder="Phone Number"
          />
          {formik.touched.parentPhone && formik.errors.parentPhone ? (
            <p className="text-red-200 text-xs mt-1">{formik.errors.parentPhone}</p>
          ) : null}
        </div>
      </div>

      {/* Parent Email */}
      <div>
        <input
          type="email"
          id="parentEmail"
          {...formik.getFieldProps('parentEmail')}
          disabled={mutation.isPending}
          className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50"
          placeholder="Email Address"
        />
        {formik.touched.parentEmail && formik.errors.parentEmail ? (
          <p className="text-red-200 text-xs mt-1">{formik.errors.parentEmail}</p>
        ) : null}
      </div>

      {/* Message */}
      <div>
        <textarea
          id="message"
          {...formik.getFieldProps('message')}
          disabled={mutation.isPending}
          rows={4}
          className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-lg text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 disabled:opacity-50 resize-none"
          placeholder="Tell us about your child's interests, academic background, or any special considerations..."
        />
        {formik.touched.message && formik.errors.message ? (
          <p className="text-red-200 text-xs mt-1">{formik.errors.message}</p>
        ) : null}
      </div>

      {/* Submit Button */}
      <button
        type="submit"
        disabled={mutation.isPending}
        className="w-full bg-white text-primary-600 hover:bg-gray-100 disabled:opacity-50 transition-colors duration-300 py-3 font-semibold rounded-lg"
      >
        {mutation.isPending ? 'Submitting...' : 'Submit Application'}
      </button>
    </form>
  );
};

export default AdmissionsForm;
