'use client';

import React from 'react';
import { useFormik } from 'formik';
import { Send, AlertCircle, CheckCircle } from 'lucide-react';
import { CONTACT_SCHEMA, CONTACT_SUBJECTS } from '@/lib/schemas';
import { useContactFormSubmission, type ContactFormData } from '@/hooks';

interface ContactFormProps {
  className?: string;
  onSuccess?: () => void;
}

const ContactForm: React.FC<ContactFormProps> = ({ className, onSuccess }) => {
  const [statusMessage, setStatusMessage] = React.useState<{
    type: 'success' | 'error' | null;
    message: string;
  }>({
    type: null,
    message: '',
  });

  const initialValues: ContactFormData = {
    name: '',
    email: '',
    phone: '',
    subject: '',
    message: '',
  };

  // Custom React Query hook for form submission
  const mutation = useContactFormSubmission(
    (data) => {
      setStatusMessage({
        type: 'success',
        message: data.message || 'Thank you! Your message has been sent successfully. We will contact you soon.',
      });

      formik.resetForm();

      if (onSuccess) {
        onSuccess();
      }

      // Clear success message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    },
    (error) => {
      setStatusMessage({
        type: 'error',
        message: error.message || 'An error occurred. Please try again.',
      });

      // Clear error message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    }
  );

  const formik = useFormik({
    initialValues,
    validationSchema: CONTACT_SCHEMA,
    validateOnChange: true,
    validateOnBlur: true,
    onSubmit: (values) => {
      setStatusMessage({ type: null, message: '' });
      mutation.mutate(values);
    },
  });

  return (
    <div className={className}>
      {/* Status Messages */}
      {statusMessage.type && (
        <div
          className={`mb-6 p-4 rounded-lg flex items-start gap-3 ${
            statusMessage.type === 'success'
              ? 'bg-green-50 border border-green-200'
              : 'bg-red-50 border border-red-200'
          }`}
        >
          {statusMessage.type === 'success' ? (
            <>
              <CheckCircle className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
              <p className="text-green-700 text-sm">{statusMessage.message}</p>
            </>
          ) : (
            <>
              <AlertCircle className="w-5 h-5 text-red-600 mt-0.5 flex-shrink-0" />
              <p className="text-red-700 text-sm">{statusMessage.message}</p>
            </>
          )}
        </div>
      )}

      <form onSubmit={formik.handleSubmit} className="space-y-6">
        {/* Name Field (Single) */}
        <div>
          <label htmlFor="name" className="block text-sm font-semibold text-gray-900 mb-2">
            Full Name *
          </label>
          <input
            type="text"
            id="name"
            {...formik.getFieldProps('name')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="Your full name"
          />
          {formik.touched.name && formik.errors.name ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.name}</p>
          ) : null}
        </div>

        {/* Email Field */}
        <div>
          <label htmlFor="email" className="block text-sm font-semibold text-gray-900 mb-2">
            Email Address *
          </label>
          <input
            type="email"
            id="email"
            {...formik.getFieldProps('email')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="your.email@example.com"
          />
          {formik.touched.email && formik.errors.email ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.email}</p>
          ) : null}
        </div>

        {/* Phone Field */}
        <div>
          <label htmlFor="phone" className="block text-sm font-semibold text-gray-900 mb-2">
            Phone Number
          </label>
          <input
            type="tel"
            id="phone"
            {...formik.getFieldProps('phone')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="+93 XX XXX XXXX"
          />
          {formik.touched.phone && formik.errors.phone ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.phone}</p>
          ) : null}
        </div>

        {/* Subject Field */}
        <div>
          <label htmlFor="subject" className="block text-sm font-semibold text-gray-900 mb-2">
            Subject *
          </label>
          <select
            id="subject"
            {...formik.getFieldProps('subject')}
            disabled={mutation.isPending}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
          >
            <option value="">Select a subject</option>
            {CONTACT_SUBJECTS.map((subject) => (
              <option key={subject.value} value={subject.value}>
                {subject.label}
              </option>
            ))}
          </select>
          {formik.touched.subject && formik.errors.subject ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.subject}</p>
          ) : null}
        </div>

        {/* Message Field */}
        <div>
          <label htmlFor="message" className="block text-sm font-semibold text-gray-900 mb-2">
            Message *
          </label>
          <textarea
            id="message"
            {...formik.getFieldProps('message')}
            disabled={mutation.isPending}
            rows={6}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors resize-vertical disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="Please describe your inquiry or message..."
          />
          {formik.touched.message && formik.errors.message ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.message}</p>
          ) : null}
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={mutation.isPending || !formik.dirty}
          className={`w-full font-bold py-4 px-8 rounded-lg transition-all duration-300 flex items-center justify-center gap-3 ${
            mutation.isPending || !formik.dirty
              ? 'bg-gray-400 text-white cursor-not-allowed'
              : 'bg-gradient-to-r from-primary-600 to-secondary-600 hover:from-primary-700 hover:to-secondary-700 text-white hover:scale-105 shadow-lg hover:shadow-xl'
          }`}
        >
          {mutation.isPending ? (
            <>
              <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
              Sending...
            </>
          ) : (
            <>
              <Send className="w-5 h-5" />
              Send Message
            </>
          )}
        </button>
      </form>
    </div>
  );
};

export default ContactForm;
