'use client';

import React from 'react';
import { useFormik } from 'formik';
import { Heart, AlertCircle, CheckCircle } from 'lucide-react';
import * as Yup from 'yup';
import { useDonationSubmission, type DonationFormData } from '@/hooks';

interface DonationFormProps {
  className?: string;
  onSuccess?: () => void;
}

// Validation schema for donation
const DONATION_SCHEMA = Yup.object().shape({
  donorName: Yup.string().required('Name is required').min(2).max(100),
  email: Yup.string().email('Invalid email').required('Email is required'),
  phone: Yup.string().optional().matches(/^[\d\s\-\+\(\)]*$/, 'Valid phone required'),
  amount: Yup.number().required('Amount is required').positive('Amount must be positive').min(1, 'Minimum donation is $1'),
  currency: Yup.string().required('Currency is required'),
  category: Yup.string().required('Please select a donation category'),
  message: Yup.string().optional().max(5000),
  paymentMethod: Yup.string().required('Payment method is required'),
  anonymous: Yup.boolean(),
});

const DONATION_CATEGORIES = [
  { value: 'general', label: 'General Support' },
  { value: 'scholarships', label: 'Scholarships' },
  { value: 'facilities', label: 'Facilities & Infrastructure' },
  { value: 'technology', label: 'Technology & Resources' },
  { value: 'programs', label: 'Special Programs' },
  { value: 'emergency', label: 'Emergency Fund' },
];

const CURRENCIES = [
  { value: 'USD', label: 'USD ($)' },
  { value: 'AFN', label: 'AFN (؋)' },
  { value: 'EUR', label: 'EUR (€)' },
];

const PAYMENT_METHODS = [
  { value: 'credit_card', label: 'Credit Card' },
  { value: 'paypal', label: 'PayPal' },
  { value: 'bank_transfer', label: 'Bank Transfer' },
  { value: 'cryptocurrency', label: 'Cryptocurrency' },
];

const DonationForm: React.FC<DonationFormProps> = ({ className, onSuccess }) => {
  const [statusMessage, setStatusMessage] = React.useState<{
    type: 'success' | 'error' | null;
    message: string;
  }>({
    type: null,
    message: '',
  });

  const initialValues: DonationFormData = {
    donorName: '',
    email: '',
    phone: '',
    amount: 0,
    currency: 'USD',
    category: '',
    message: '',
    paymentMethod: '',
    anonymous: false,
  };

  // Custom React Query hook for donation submission
  const mutation = useDonationSubmission(
    (data) => {
      setStatusMessage({
        type: 'success',
        message: data.message || 'Thank you for your generous donation! We will contact you soon with a receipt.',
      });

      formik.resetForm();

      if (onSuccess) {
        onSuccess();
      }

      // Clear success message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    },
    (error) => {
      setStatusMessage({
        type: 'error',
        message: error.message || 'An error occurred. Please try again.',
      });

      // Clear error message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    }
  );

  const formik = useFormik({
    initialValues,
    validationSchema: DONATION_SCHEMA,
    validateOnChange: true,
    validateOnBlur: true,
    onSubmit: (values) => {
      setStatusMessage({ type: null, message: '' });
      mutation.mutate(values);
    },
  });

  return (
    <div className={className}>
      {/* Status Messages */}
      {statusMessage.type && (
        <div
          className={`mb-6 p-4 rounded-lg flex items-start gap-3 ${
            statusMessage.type === 'success'
              ? 'bg-green-50 border border-green-200'
              : 'bg-red-50 border border-red-200'
          }`}
        >
          {statusMessage.type === 'success' ? (
            <>
              <CheckCircle className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
              <p className="text-green-700 text-sm">{statusMessage.message}</p>
            </>
          ) : (
            <>
              <AlertCircle className="w-5 h-5 text-red-600 mt-0.5 flex-shrink-0" />
              <p className="text-red-700 text-sm">{statusMessage.message}</p>
            </>
          )}
        </div>
      )}

      <form onSubmit={formik.handleSubmit} className="space-y-6">
        {/* Donation Amount & Currency */}
        <div className="grid grid-cols-1 sm:grid-cols-3 gap-6">
          <div className="sm:col-span-2">
            <label htmlFor="amount" className="block text-sm font-semibold text-gray-900 mb-2">
              Donation Amount *
            </label>
            <div className="flex gap-2">
              <select
                id="currency"
                {...formik.getFieldProps('currency')}
                disabled={mutation.isPending}
                className="px-3 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100"
              >
                {CURRENCIES.map((curr) => (
                  <option key={curr.value} value={curr.value}>
                    {curr.label}
                  </option>
                ))}
              </select>
              <input
                type="number"
                id="amount"
                {...formik.getFieldProps('amount')}
                disabled={mutation.isPending}
                className="flex-1 px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
                placeholder="0.00"
                step="0.01"
              />
            </div>
            {formik.touched.amount && formik.errors.amount ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.amount}</p>
            ) : null}
          </div>

          {/* Donation Category */}
          <div>
            <label htmlFor="category" className="block text-sm font-semibold text-gray-900 mb-2">
              Category *
            </label>
            <select
              id="category"
              {...formik.getFieldProps('category')}
              disabled={mutation.isPending}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100"
            >
              <option value="">Select category</option>
              {DONATION_CATEGORIES.map((cat) => (
                <option key={cat.value} value={cat.value}>
                  {cat.label}
                </option>
              ))}
            </select>
            {formik.touched.category && formik.errors.category ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.category}</p>
            ) : null}
          </div>
        </div>

        {/* Donor Name & Email */}
        <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
          <div>
            <label htmlFor="donorName" className="block text-sm font-semibold text-gray-900 mb-2">
              Full Name *
            </label>
            <input
              type="text"
              id="donorName"
              {...formik.getFieldProps('donorName')}
              disabled={mutation.isPending}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
              placeholder="Your full name"
            />
            {formik.touched.donorName && formik.errors.donorName ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.donorName}</p>
            ) : null}
          </div>

          <div>
            <label htmlFor="email" className="block text-sm font-semibold text-gray-900 mb-2">
              Email Address *
            </label>
            <input
              type="email"
              id="email"
              {...formik.getFieldProps('email')}
              disabled={mutation.isPending}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
              placeholder="your.email@example.com"
            />
            {formik.touched.email && formik.errors.email ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.email}</p>
            ) : null}
          </div>
        </div>

        {/* Phone & Payment Method */}
        <div className="grid grid-cols-1 sm:grid-cols-2 gap-6">
          <div>
            <label htmlFor="phone" className="block text-sm font-semibold text-gray-900 mb-2">
              Phone (Optional)
            </label>
            <input
              type="tel"
              id="phone"
              {...formik.getFieldProps('phone')}
              disabled={mutation.isPending}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
              placeholder="+93 XX XXX XXXX"
            />
            {formik.touched.phone && formik.errors.phone ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.phone}</p>
            ) : null}
          </div>

          <div>
            <label htmlFor="paymentMethod" className="block text-sm font-semibold text-gray-900 mb-2">
              Payment Method *
            </label>
            <select
              id="paymentMethod"
              {...formik.getFieldProps('paymentMethod')}
              disabled={mutation.isPending}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100"
            >
              <option value="">Select payment method</option>
              {PAYMENT_METHODS.map((method) => (
                <option key={method.value} value={method.value}>
                  {method.label}
                </option>
              ))}
            </select>
            {formik.touched.paymentMethod && formik.errors.paymentMethod ? (
              <p className="text-red-600 text-sm mt-1">{formik.errors.paymentMethod}</p>
            ) : null}
          </div>
        </div>

        {/* Message */}
        <div>
          <label htmlFor="message" className="block text-sm font-semibold text-gray-900 mb-2">
            Message (Optional)
          </label>
          <textarea
            id="message"
            {...formik.getFieldProps('message')}
            disabled={mutation.isPending}
            rows={4}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors resize-vertical disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="Tell us why this donation is important to you..."
          />
          {formik.touched.message && formik.errors.message ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.message}</p>
          ) : null}
        </div>

        {/* Anonymous Checkbox */}
        <div className="flex items-center gap-3">
          <input
            type="checkbox"
            id="anonymous"
            {...formik.getFieldProps('anonymous')}
            disabled={mutation.isPending}
            className="w-4 h-4 text-primary-600 rounded focus:ring-2 focus:ring-primary-500 cursor-pointer disabled:cursor-not-allowed"
          />
          <label htmlFor="anonymous" className="text-sm text-gray-700 cursor-pointer">
            Make this donation anonymous
          </label>
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={mutation.isPending || !formik.dirty}
          className={`w-full font-bold py-4 px-8 rounded-lg transition-all duration-300 flex items-center justify-center gap-3 ${
            mutation.isPending || !formik.dirty
              ? 'bg-gray-400 text-white cursor-not-allowed'
              : 'bg-gradient-to-r from-primary-600 to-secondary-600 hover:from-primary-700 hover:to-secondary-700 text-white hover:scale-105 shadow-lg hover:shadow-xl'
          }`}
        >
          {mutation.isPending ? (
            <>
              <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
              Processing...
            </>
          ) : (
            <>
              <Heart className="w-5 h-5" />
              Make a Donation
            </>
          )}
        </button>
      </form>
    </div>
  );
};

export default DonationForm;
