'use client';

import React from 'react';
import { useFormik } from 'formik';
import { Mail, CheckCircle, AlertCircle } from 'lucide-react';
import * as Yup from 'yup';
import { useNewsletterSubscription, type NewsletterFormData } from '@/hooks';

interface NewsletterFormProps {
  className?: string;
  onSuccess?: () => void;
  compact?: boolean;
  setIsSubscribed?: (subscribed: boolean) => void;
}

// Validation schema for newsletter
const NEWSLETTER_SCHEMA = Yup.object().shape({
  email: Yup.string().email('Invalid email address').required('Email is required'),
  firstName: Yup.string().optional().max(100, 'Name too long'),
});

const NewsletterForm: React.FC<NewsletterFormProps> = ({ className, onSuccess, compact = false, setIsSubscribed }) => {
  const [statusMessage, setStatusMessage] = React.useState<{
    type: 'success' | 'error' | null;
    message: string;
  }>({
    type: null,
    message: '',
  });

  const initialValues: NewsletterFormData = {
    email: '',
    firstName: '',
  };

  // Custom React Query hook for newsletter subscription
  const mutation = useNewsletterSubscription(
    (data) => {
      setStatusMessage({
        type: 'success',
        message: data.message || 'Thank you for subscribing! Check your email for confirmation.',
      });

      formik.resetForm();

      if (onSuccess) {
        setIsSubscribed?.(true);
        onSuccess();
      }

      // Clear success message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    },
    (error) => {
      setStatusMessage({
        type: 'error',
        message: error.message || 'Failed to subscribe. Please try again.',
      });

      // Clear error message after 5 seconds
      setTimeout(() => {
        setStatusMessage({ type: null, message: '' });
      }, 5000);
    }
  );

  const formik = useFormik({
    initialValues,
    validationSchema: NEWSLETTER_SCHEMA,
    validateOnChange: true,
    validateOnBlur: true,
    onSubmit: (values) => {
      setStatusMessage({ type: null, message: '' });
      mutation.mutate(values);
    },
  });

  if (compact) {
    // Compact horizontal layout for footer/sidebar
    return (
      <div className={className}>
        {statusMessage.type && (
          <div
            className={`mb-3 p-2 rounded text-xs ${
              statusMessage.type === 'success'
                ? 'bg-green-100 text-green-700'
                : 'bg-red-100 text-red-700'
            }`}
          >
            {statusMessage.message}
          </div>
        )}

        <form onSubmit={formik.handleSubmit} className="flex flex-col gap-2">
          <input
            type="email"
            id="email"
            {...formik.getFieldProps('email')}
            disabled={mutation.isPending}
            className="px-3 py-2 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-primary-500 focus:border-primary-500 transition-colors disabled:bg-gray-100 disabled:cursor-not-allowed"
            placeholder="your.email@example.com"
          />
          {formik.touched.email && formik.errors.email ? (
            <p className="text-red-600 text-xs">{formik.errors.email}</p>
          ) : null}

          <button
            type="submit"
            disabled={mutation.isPending || !formik.dirty}
            className="bg-primary-600 hover:bg-primary-700 disabled:bg-gray-400 text-white font-semibold py-2 px-4 rounded-lg transition-colors disabled:cursor-not-allowed text-sm"
          >
            {mutation.isPending ? 'Subscribing...' : 'Subscribe'}
          </button>
        </form>
      </div>
    );
  }

  // Full layout for dedicated newsletter pages/sections
  return (
    <div className={className}>
      {/* Status Messages */}
      {statusMessage.type && (
        <div
          className={`mb-6 p-4 rounded-lg flex items-start gap-3 ${
            statusMessage.type === 'success'
              ? 'bg-green-50 border border-green-200'
              : 'bg-red-50 border border-red-200'
          }`}
        >
          {statusMessage.type === 'success' ? (
            <>
              <CheckCircle className="w-5 h-5 text-green-600 mt-0.5 flex-shrink-0" />
              <p className="text-green-700 text-sm">{statusMessage.message}</p>
            </>
          ) : (
            <>
              <AlertCircle className="w-5 h-5 text-red-600 mt-0.5 flex-shrink-0" />
              <p className="text-red-700 text-sm">{statusMessage.message}</p>
            </>
          )}
        </div>
      )}

      <form onSubmit={formik.handleSubmit} className="space-y-6">
        {/* Email Field */}
        <div>
          <label htmlFor="email" className="block text-sm font-semibold text-gray-900 mb-2">
            Email Address *
          </label>
          <input
            type="email"
            id="email"
            {...formik.getFieldProps('email')}
            disabled={mutation.isPending}
            className="w-full px-4 py-4 border border-gray-200 rounded-xl focus:border-primary-500 focus:ring-2 focus:ring-primary-100 transition-all duration-200 text-lg disabled:cursor-not-allowed"
            placeholder="your.email@example.com"
          />
          {formik.touched.email && formik.errors.email ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.email}</p>
          ) : null}
        </div>

        {/* Name Field (Optional) */}
        <div>
          <label htmlFor="firstName" className="block text-sm font-semibold text-gray-900 mb-2">
            First Name (Optional)
          </label>
          <input
            type="text"
            id="firstName"
            {...formik.getFieldProps('firstName')}
            disabled={mutation.isPending}
            className="w-full px-4 py-4 border border-gray-200 rounded-xl focus:border-primary-500 focus:ring-2 focus:ring-primary-100 transition-all duration-200 text-lg disabled:cursor-not-allowed"
            placeholder="Your first name"
          />
          {formik.touched.firstName && formik.errors.firstName ? (
            <p className="text-red-600 text-sm mt-1">{formik.errors.firstName}</p>
          ) : null}
        </div>

        {/* Subscribe Button */}
        <button
          type="submit"
          disabled={mutation.isPending || !formik.dirty}
          className={`bg-gradient-to-r w-full font-bold py-4 px-8 rounded-lg transition-all duration-300 flex items-center justify-center gap-3 ${
            mutation.isPending || !formik.dirty
              ? 'disabled:from-gray-300 disabled:to-gray-400  text-white cursor-not-allowed'
              : 'w-full from-primary-600 to-secondary-600 hover:from-primary-700 hover:to-secondary-700 text-white py-4 px-6 rounded-xl font-semibold text-lg transition-all duration-200 flex items-center justify-center gap-2 shadow-lg hover:shadow-xl disabled:cursor-not-allowed'
          }`}
        >
          {mutation.isPending ? (
            <>
              <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
              Subscribing...
            </>
          ) : (
            <>
              <Mail className="w-5 h-5" />
              Subscribe to Newsletter
            </>
          )}
        </button>
      </form>
    </div>
  );
};

export default NewsletterForm;
