"use client"
import React from "react";
import Link from "next/link";
import {
  FiMapPin,
  FiPhone,
  FiMail,
} from "react-icons/fi"; 
import { FaFacebookF, FaXTwitter , 
  FaInstagram,
  FaYoutube } from "react-icons/fa6";
import { usePublicSiteSettings } from "@/hooks/useSiteSettings";
import NewsletterForm from "@/components/common/forms/NewsletterForm";

type SocialLink = {
  href: string;
  label: string;
  icon: React.ReactNode;
};

const FooterLinkColumn: React.FC<{
  title: string;
  links: { href: string; label: string }[];
}> = ({ title, links }) => (
  <div>
    <h3 className="text-lg font-semibold text-white mb-4">{title}</h3>
    <ul className="space-y-3">
      {links.map((l, i) => (
        <li key={i}>
          <Link
            href={l.href}
            className="text-gray-300 hover:text-primary-400 transition-colors"
          >
            {l.label}
          </Link>
        </li>
      ))}
    </ul>
  </div>
);

const SocialIcon: React.FC<{
  href: string;
  label: string;
  children: React.ReactNode;
}> = ({ href, label, children }) => (
  <a
    href={href}
    aria-label={label}
    target="_blank"
    rel="noopener noreferrer"
    className="w-10 h-10 bg-gray-800 rounded-lg flex items-center justify-center hover:bg-primary-600 transition-colors"
  >
    {children}
  </a>
);

const Footer: React.FC = () => {
  const { data: settings } = usePublicSiteSettings([
    'logo_url',
    'logo_text',
    'logo_initials',
    'show_logo_text',
    'contact_address_line1',
    'contact_address_line2',
    'contact_phone_primary',
    'contact_email',
    'social_facebook',
    'social_linkedin',
    'social_whatsapp'
  ]);
  
  const logoText = settings?.logo_text || 'Setaragan Zirak';
  const logoInitials = settings?.logo_initials || 'SZ';
  const showLogoText = settings?.show_logo_text !== 'false';
  const logoUrl = settings?.logo_url;

  // Contact information with fallbacks
  const addressLine1 = settings?.contact_address_line1 || 'Market Safid Street, 2nd Phase,';
  const addressLine2 = settings?.contact_address_line2 || 'Omid-e-Sabz Township Darullaman, Kabul, Afghanistan';
  const phone = settings?.contact_phone_primary || '+93 (0) 123 456 789';
  const email = settings?.contact_email || 'info@setaraganzirak.edu.af';

  // Social links with dynamic URLs
  const dynamicSocialLinks: SocialLink[] = [
    { 
      href: settings?.social_facebook || "#", 
      label: "Facebook", 
      icon: <FaFacebookF /> 
    },
    { 
      href: settings?.social_linkedin || "#", 
      label: "LinkedIn", 
      icon: <FaXTwitter  /> 
    },
    { 
      href: settings?.social_whatsapp || "#", 
      label: "WhatsApp", 
      icon: <FaInstagram /> 
    },
    { 
      href: "#", 
      label: "YouTube", 
      icon: <FaYoutube /> 
    },
  ];
  
  return (
    <footer className="bg-gray-900 text-white">
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {/* Logo & Description */}
          <div className="lg:col-span-1">
            <div className="flex items-center gap-x-3 mb-4">
              {logoUrl ? (
                <img src={logoUrl} alt={logoText} className="h-10 object-contain" />
              ) : (
                <div className="w-10 h-10 bg-primary-600 rounded-lg flex items-center justify-center text-white font-bold text-lg">
                  {logoInitials}
                </div>
              )}
              {showLogoText && (
                <div className="flex items-baseline">
                  <span className="text-xl font-bold text-white">{logoText.split(' ')[0]}</span>
                  {logoText.split(' ')[1] && (
                    <span className="text-xl font-bold text-primary-400 ml-1">
                      {logoText.split(' ').slice(1).join(' ')}
                    </span>
                  )}
                </div>
              )}
            </div>
            <p className="text-gray-300 leading-relaxed mb-6 max-w-sm">
              Empowering generations through quality education and fostering a brighter future for our community.
            </p>
            <div className="flex gap-x-4">
              {dynamicSocialLinks.map((s, i) => (
                <SocialIcon
                  key={i}
                  href={s.href}
                  label={s.label}
                >
                  {s.icon}
                </SocialIcon>
              ))}
            </div>
          </div>

          {/* Link columns */}
          <div className="lg:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-8">
            <FooterLinkColumn
              title="About School"
              links={[
                { href: "/about", label: "About Us" },
                { href: "/about#our-story", label: "Our Story" },
                { href: "/about#mission-values", label: "Mission & Values" },
                { href: "/about#timeline", label: "Our History" },
                { href: "/about#leadership", label: "Leadership Team" },
                { href: "/contact", label: "Contact Us" },
              ]}
            />
            <FooterLinkColumn
              title="Academic Programs"
              links={[
                { href: "/students", label: "Student Stories" },
                { href: "/services", label: "Services & Programs" },
                { href: "/admissions", label: "Admissions" },
                { href: "/facilities", label: "Facilities" },
                { href: "/team", label: "Our Team" },
                { href: "/online-school", label: "Online School" },
              ]}
            />
          </div>

          {/* Contact & Newsletter */}
          <div>
            <h3 className="text-lg font-semibold text-white mb-4">
              Contact Info
            </h3>
            <div className="space-y-4">
              <div className="flex items-start gap-x-3">
                <FiMapPin className="h-5 w-5 text-primary-400 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-300">
                    {addressLine1} <br />
                    {addressLine2}
                  </p>
                </div>
              </div>
              <div className="flex items-center gap-x-3">
                <FiPhone className="h-5 w-5 text-primary-400 flex-shrink-0" />
                <p className="text-gray-300">{phone}</p>
              </div>
              <div className="flex items-center gap-x-3">
                <FiMail className="h-5 w-5 text-primary-400 flex-shrink-0" />
                <p className="text-gray-300">{email}</p>
              </div>
            </div>

            <div className="mt-6">
              <h4 className="text-md font-semibold text-white mb-3">
                Stay Updated
              </h4>
              <NewsletterForm compact={true} />
            </div>
          </div>
        </div>

        <div className="border-t border-gray-800 mt-12 pt-8">
          <div className="flex flex-col md:flex-row justify-between items-center gap-y-4 md:gap-y-0">
            <div className="flex flex-col md:flex-row items-center gap-y-2 md:gap-y-0 md:gap-x-6">
              <p className="text-gray-400 text-sm">
                © {new Date().getFullYear()} Setaragan Zirak School. All rights reserved.
              </p>
              <div className="flex gap-x-6">
                <Link
                  href="/privacy"
                  className="text-gray-400 hover:text-primary-400 text-sm transition-colors"
                >
                  Privacy Policy
                </Link>
                <Link
                  href="/terms"
                  className="text-gray-400 hover:text-primary-400 text-sm transition-colors"
                >
                  Terms of Service
                </Link>
                <Link
                  href="/accessibility"
                  className="text-gray-400 hover:text-primary-400 text-sm transition-colors"
                >
                  Accessibility
                </Link>
              </div>
            </div>

            <div className="flex items-center gap-x-2 text-gray-400 text-sm">
              <span>Made with</span>
              <span className="text-red-500">❤️</span>
              <span>for education</span>
            </div>
          </div>
        </div>
      </div>
    </footer>
  );
};

export default Footer;
