'use client';

import React, { useState, useEffect, useCallback, useMemo } from 'react';
import Link from 'next/link';
import { usePathname } from 'next/navigation';
import { Menu, X, ChevronDown } from 'lucide-react';
import { usePublicSiteSettings } from '@/hooks/useSiteSettings';

// Navigation structure definition
const NAV_ITEMS = [
  { 
    label: 'Home', 
    href: '/', 
    type: 'link' as const 
  },
  { 
    label: 'About', 
    type: 'dropdown' as const,
    paths: ['/about'],
    items: [
      { label: 'About Us', href: '/about' },
      { label: 'Our Story', href: '/about#our-story' },
      { label: 'Mission & Values', href: '/about#mission-values' },
      { label: 'Our History', href: '/about#timeline' },
      { label: 'Leadership Team', href: '/about#leadership' },
      { label: 'Community Impact', href: '/about#empowerment' },
      { label: 'Student Testimonials', href: '/about#testimonials' }
    ]
  },
  { 
    label: 'Academic', 
    type: 'dropdown' as const,
    paths: ['/students', '/services', '/admissions'],
    items: [
      { label: 'Student Stories', href: '/students' },
      { label: 'Services & Programs', href: '/services' },
      { label: 'Admissions', href: '/admissions' }
    ]
  },
  { 
    label: 'School', 
    type: 'dropdown' as const,
    paths: ['/facilities', '/team'],
    items: [
      { label: 'Facilities', href: '/facilities' },
      { label: 'Our Team', href: '/team' }
    ]
  },
  { 
    label: 'Online School', 
    href: '/online-school', 
    type: 'link' as const 
  },
  { 
    label: 'News', 
    href: '/news', 
    type: 'link' as const 
  },
  { 
    label: 'Contact', 
    href: '/contact', 
    type: 'link' as const 
  }
];

const Header = () => {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [openDropdown, setOpenDropdown] = useState<string | null>(null);
  const [mobileOpenDropdown, setMobileOpenDropdown] = useState<string | null>(null);
  
  const pathname = usePathname();

  // Fetch logo settings
  const { data: settings } = usePublicSiteSettings(['logo_url', 'logo_text', 'logo_initials', 'show_logo_text']);
  
  const logoText = settings?.logo_text || 'Setaragan Zirak';
  const logoInitials = settings?.logo_initials || 'SZ';
  const showLogoText = settings?.show_logo_text !== 'false';
  const logoUrl = settings?.logo_url;

  // Close all dropdowns when menu closes
  useEffect(() => {
    if (!isMenuOpen) {
      setMobileOpenDropdown(null);
    }
  }, [isMenuOpen]);

  // Prevent body scrolling when mobile menu is open
  useEffect(() => {
    if (isMenuOpen) {
      document.body.style.overflow = 'hidden';
      document.body.style.position = 'fixed';
      document.body.style.width = '100%';
    } else {
      document.body.style.overflow = '';
      document.body.style.position = '';
      document.body.style.width = '';
    }

    return () => {
      document.body.style.overflow = '';
      document.body.style.position = '';
      document.body.style.width = '';
    };
  }, [isMenuOpen]);

  // Enhanced helper function to check if link is active (including hash fragments)
const isActiveLink = useCallback((href: string) => {
  if (href === '/') return pathname === '/';
  
  // Handle hash fragments
  if (href.includes('#')) {
    const [path, hash] = href.split('#');
    if (pathname === path) {
      // Check if we're on the same page and the hash matches
      if (typeof window !== 'undefined') {
        return window.location.hash === `#${hash}`;
      }
      // Fallback for SSR
      return true;
    }
    return false;
  }
  
  return pathname.startsWith(href);
}, [pathname]);

  // Helper function to check if dropdown section is active
  const isDropdownActive = useCallback((paths: string[]) => {
    return paths.some(path => isActiveLink(path));
  }, [isActiveLink]);

  // Toggle mobile dropdown with accordion behavior
  const toggleMobileDropdown = useCallback((label: string) => {
    setMobileOpenDropdown(mobileOpenDropdown === label ? null : label);
  }, [mobileOpenDropdown]);

  // Close mobile menu and reset state
  const closeMobileMenu = useCallback(() => {
    setIsMenuOpen(false);
    setMobileOpenDropdown(null);
  }, []);

  // Render desktop navigation items
  const renderDesktopNav = useMemo(() => (
    <>
      {NAV_ITEMS.map((item) => {
        if (item.type === 'link') {
          return (
            <Link
              key={item.href}
              href={item.href}
              className={`relative px-4 py-2 font-medium transition-all duration-300 rounded-lg group overflow-hidden ${
                isActiveLink(item.href) 
                  ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                  : 'text-gray-700 hover:text-primary-600'
              }`}
              onClick={closeMobileMenu}
            >
              <span className="relative z-10">{item.label}</span>
              {!isActiveLink(item.href) && (
                <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 scale-x-0 group-hover:scale-x-100 transition-transform duration-300 origin-left"></div>
              )}
            </Link>
          );
        }

        return (
          <div
            key={item.label}
            className="relative group"
            onMouseEnter={() => setOpenDropdown(item.label)}
            onMouseLeave={() => setOpenDropdown(null)}
          >
            <button 
              className={`relative flex items-center px-4 py-2 font-medium transition-all duration-300 rounded-lg group overflow-hidden ${
                isDropdownActive(item.paths) 
                  ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                  : 'text-gray-700 hover:text-primary-600'
              }`}
              aria-expanded={openDropdown === item.label}
              aria-haspopup="true"
            >
              <span className="relative z-10 flex items-center">
                {item.label}
                <ChevronDown className="ml-1 h-4 w-4 transition-transform duration-200 group-hover:rotate-180" />
              </span>
              {!isDropdownActive(item.paths) && (
                <div className="absolute inset-0 bg-gradient-to-r from-primary-50 to-secondary-50 scale-x-0 group-hover:scale-x-100 transition-transform duration-300 origin-left"></div>
              )}
            </button>
            
            <div className={`absolute top-full left-0 mt-3 w-64 bg-white/95 backdrop-blur-lg rounded-xl shadow-2xl border border-gray-100/50 transform transition-all duration-300 z-50 ${
              openDropdown === item.label ? 'opacity-100 visible translate-y-0 scale-100' : 'opacity-0 invisible -translate-y-2 scale-95'
            }`}>
              <div className="py-3">
                {item.items.map((subItem) => (
                  <Link
                    key={subItem.href}
                    href={subItem.href}
                    className={`flex items-center px-4 py-3 transition-all duration-200 ${
                      isActiveLink(subItem.href) 
                        ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                        : 'text-gray-700 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 hover:text-primary-600'
                    }`}
                    onClick={closeMobileMenu}
                  >
                    <span className="font-medium">{subItem.label}</span>
                  </Link>
                ))}
              </div>
            </div>
          </div>
        );
      })}
    </>
  ), [openDropdown, isActiveLink, isDropdownActive, closeMobileMenu]);

  // Render mobile navigation items
  const renderMobileNav = useMemo(() => (
    <div className="px-3 py-4 space-y-2 bg-white/95 backdrop-blur-lg border-t border-gray-100/50 max-h-[80vh] overflow-y-auto overscroll-contain mobile-nav-scroll">
      {NAV_ITEMS.map((item, index) => {
        if (item.type === 'link') {
          return (
            <Link
              key={item.href}
              href={item.href}
              className={`flex items-center px-4 py-3 text-base font-medium rounded-xl transition-all duration-300 group ${
                isActiveLink(item.href) 
                  ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                  : 'text-gray-700 hover:text-primary-600 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50'
              }`}
              onClick={closeMobileMenu}
              style={{
                animationDelay: `${index * 50}ms`,
                animation: isMenuOpen ? 'slideInLeft 0.4s ease-out forwards' : 'none'
              }}
            >
              {item.label}
            </Link>
          );
        }

        return (
          <div 
            key={item.label} 
            className="group"
            style={{
              animationDelay: `${index * 50}ms`,
              animation: isMenuOpen ? 'slideInLeft 0.4s ease-out forwards' : 'none'
            }}
          >
            <button 
              onClick={() => toggleMobileDropdown(item.label)}
              className={`w-full flex items-center justify-between px-4 py-3 text-base font-medium rounded-xl cursor-pointer transition-all duration-300 ${
                isDropdownActive(item.paths) 
                  ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                  : 'text-gray-700 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 hover:text-primary-600'
              }`}
              aria-expanded={mobileOpenDropdown === item.label}
              aria-controls={`mobile-submenu-${item.label}`}
            >
              <div className="flex items-center">
                {item.label}
              </div>
              <ChevronDown className={`h-4 w-4 transition-transform duration-300 ${
                mobileOpenDropdown === item.label ? 'rotate-180' : 'rotate-0'
              }`} />
            </button>
            <div 
              id={`mobile-submenu-${item.label}`}
              className={`overflow-hidden transition-all duration-300 ${
                mobileOpenDropdown === item.label ? 'max-h-96 opacity-100' : 'max-h-0 opacity-0'
              }`}
            >
              <div className="ml-6 mt-2 space-y-1 border-l-2 border-primary-100 pl-4">
                {item.items.map((subItem) => (
                  <Link
                    key={subItem.href}
                    href={subItem.href}
                    className={`flex items-center px-4 py-2 rounded-lg transition-all duration-200 ${
                      isActiveLink(subItem.href) 
                        ? 'text-primary-600 bg-gradient-to-r from-primary-50 to-secondary-50' 
                        : 'text-gray-600 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 hover:text-primary-600'
                    }`}
                    onClick={closeMobileMenu}
                  >
                    {subItem.label}
                  </Link>
                ))}
              </div>
            </div>
          </div>
        );
      })}
    </div>
  ), [isMenuOpen, mobileOpenDropdown, isActiveLink, isDropdownActive, closeMobileMenu, toggleMobileDropdown]);

  return (
    <>
      <header className="bg-white/90 lg:bg-white/95 backdrop-blur-md shadow-xl sticky top-0 z-50 border-b border-gray-100/50">
        <nav className="max-w-8xl mx-auto px-3 sm:px-4 lg:px-6 xl:px-8">
          <div className="flex justify-between items-center h-16 sm:h-18 lg:h-20">
            {/* Logo */}
            <Link href="/" className="flex items-center space-x-3 group">
              <div className="relative w-8 h-8 sm:w-10 sm:h-10 lg:w-12 lg:h-12">
                {logoUrl ? (
                  <img 
                    src={logoUrl} 
                    alt={logoText} 
                    className="w-full h-full object-contain"
                  />
                ) : (
                  <>
                    <div className="absolute inset-0 bg-gradient-to-br from-primary-600 via-primary-500 to-secondary-600 rounded-xl flex items-center justify-center shadow-lg group-hover:shadow-2xl transition-all duration-300 group-hover:scale-110 group-hover:rotate-3">
                      <span className="text-white font-bold text-sm sm:text-base lg:text-lg drop-shadow-sm">{logoInitials}</span>
                    </div>
                    <div className="absolute inset-0 bg-gradient-to-br from-primary-400 to-secondary-400 rounded-xl opacity-0 group-hover:opacity-20 transition-opacity duration-300"></div>
                  </>
                )}
              </div>
              {showLogoText && (
                <div className="flex flex-col">
                  <span className="font-bold text-xl sm:text-2xl lg:text-3xl hidden sm:block bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent leading-tight">
                    {logoText}
                  </span>
                  <span className="font-medium text-sm block sm:hidden bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                    {logoInitials} School
                  </span>
                </div>
              )}
            </Link>

            {/* Desktop Navigation */}
            <div className="hidden lg:flex items-center space-x-1 xl:space-x-2">
              {renderDesktopNav}
            </div>

            {/* Right side buttons */}
            <div className="hidden lg:flex items-center space-x-3">
              {/* Language Selector */}
              {/* <div
                className="relative"
                onMouseEnter={() => setIsLanguageOpen(true)}
                onMouseLeave={() => setIsLanguageOpen(false)}
              >
                <button 
                  className="flex items-center px-3 py-2 text-gray-700 hover:text-primary-600 font-medium transition-all duration-300 rounded-lg border border-gray-200 hover:border-primary-300 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 group"
                  aria-expanded={isLanguageOpen}
                  aria-haspopup="true"
                >
                  <Globe className="h-4 w-4 mr-2 transition-transform duration-300 group-hover:rotate-12" />
                  <span className="text-sm">EN</span>
                  <ChevronDown className="ml-2 h-3 w-3 transition-transform duration-200 group-hover:rotate-180" />
                </button>
                
                <div className={`absolute top-full right-0 mt-2 w-40 bg-white/95 backdrop-blur-lg rounded-xl shadow-2xl border border-gray-100/50 transform transition-all duration-300 z-50 ${
                  isLanguageOpen ? 'opacity-100 visible translate-y-0 scale-100' : 'opacity-0 invisible -translate-y-2 scale-95'
                }`}>
                  <div className="py-2">
                    {LANGUAGES.map((lang) => (
                      <button 
                        key={lang.code}
                        className="flex items-center w-full text-left px-4 py-3 text-gray-700 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 hover:text-primary-600 transition-all duration-200"
                      >
                        <span className="font-medium">{lang.label}</span>
                      </button>
                    ))}
                  </div>
                </div>
              </div> */}

              {/* Donate Button */}
              <Link
                href="/donate"
                className="relative inline-flex items-center bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 text-white px-6 py-2.5 rounded-xl font-medium transition-all duration-300 hover:shadow-2xl hover:scale-105 group overflow-hidden"
              >
                <span className="relative z-10 flex items-center">
                  ❤️ Donate
                </span>
                <div className="absolute inset-0 bg-gradient-to-r from-primary-700 via-primary-600 to-secondary-700 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                <div className="absolute inset-0 bg-white/20 scale-x-0 group-hover:scale-x-100 transition-transform duration-500 origin-left"></div>
              </Link>
            </div>

            {/* Mobile/Tablet CTA + Menu */}
            <div className="flex items-center space-x-2 lg:hidden">
              {/* Mobile Donate Button */}
              <Link
                href="/donate"
                className="relative inline-flex items-center bg-gradient-to-r from-primary-600 to-secondary-600 text-white px-3 sm:px-4 py-2 rounded-lg text-xs sm:text-sm font-medium shadow-lg hover:shadow-xl transition-all duration-300 hover:scale-105 group overflow-hidden"
              >
                <span className="relative z-10 flex items-center">
                  <span className="hidden xs:inline">❤️ Donate</span>
                  <span className="inline xs:hidden text-base">💖</span>
                </span>
                <div className="absolute inset-0 bg-gradient-to-r from-primary-700 to-secondary-700 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              </Link>

              {/* Mobile Language Button */}
              {/* <button className="flex items-center space-x-1 border border-gray-300 hover:border-primary-300 text-gray-700 hover:text-primary-600 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 px-3 py-2 rounded-lg text-xs sm:text-sm font-medium transition-all duration-300 hover:scale-105 group">
                <Globe className="w-3 sm:w-4 h-3 sm:h-4 transition-transform duration-300 group-hover:rotate-12" />
                <span className="hidden xs:inline">EN</span>
              </button> */}

              {/* Mobile menu button */}
              <button
                onClick={() => setIsMenuOpen(!isMenuOpen)}
                className="relative p-2.5 text-gray-700 hover:text-primary-600 hover:bg-gradient-to-r hover:from-primary-50 hover:to-secondary-50 rounded-lg transition-all duration-300 hover:scale-105 group"
                aria-label={isMenuOpen ? "Close menu" : "Open menu"}
                aria-expanded={isMenuOpen}
              >
                <div className="relative">
                  {isMenuOpen ? (
                    <X className="w-5 h-5 sm:w-6 sm:h-6 transition-transform duration-300 rotate-90" />
                  ) : (
                    <Menu className="w-5 h-5 sm:w-6 sm:h-6 transition-transform duration-300 group-hover:scale-110" />
                  )}
                </div>
              </button>
            </div>
          </div>

          {/* Mobile Navigation */}
          <div className={`lg:hidden transition-all duration-500 ease-in-out overflow-hidden ${
            isMenuOpen ? "max-h-[90vh] opacity-100" : "max-h-0 opacity-0"
          }`}>
            {renderMobileNav}
          </div>
        </nav>
      </header>
      
      {/* Mobile Menu Overlay */}
      {isMenuOpen && (
        <div 
          className="lg:hidden fixed inset-0 bg-black/30 backdrop-blur-sm z-40 transition-all duration-300"
          onClick={closeMobileMenu}
          aria-hidden="true"
        />
      )}

      <style jsx>{`
        @keyframes slideInLeft {
          from {
            opacity: 0;
            transform: translateX(-30px);
          }
          to {
            opacity: 1;
            transform: translateX(0);
          }
        }
        
        /* Custom scrollbar for mobile menu */
        .mobile-nav-scroll::-webkit-scrollbar {
          width: 4px;
        }
        
        .mobile-nav-scroll::-webkit-scrollbar-track {
          background: transparent;
        }
        
        .mobile-nav-scroll::-webkit-scrollbar-thumb {
          background: rgba(99, 102, 241, 0.3);
          border-radius: 2px;
        }
        
        .mobile-nav-scroll::-webkit-scrollbar-thumb:hover {
          background: rgba(99, 102, 241, 0.5);
        }
      `}</style>
    </>
  );
};

export default Header;