import React, { Suspense } from 'react';
import { ChevronRight } from 'lucide-react';
import Link from 'next/link';

// Loading Components
export const SkeletonLoader: React.FC<{ className?: string }> = ({ className = "" }) => (
  <div className={`animate-pulse bg-gray-200 rounded ${className}`} />
);

// Breadcrumb Component
interface Breadcrumb {
  name: string;
  href?: string;
}

interface BreadcrumbsProps {
  items: Breadcrumb[];
}

export const Breadcrumbs: React.FC<BreadcrumbsProps> = ({ items }) => (
  <nav className="flex items-center space-x-2 text-sm mb-6">
    {items.map((item, index) => (
      <React.Fragment key={index}>
        {index > 0 && <ChevronRight className="h-4 w-4 text-gray-400" />}
        {item.href ? (
          <Link
            href={item.href}
            className="text-gray-500 hover:text-primary-600 transition-colors"
          >
            {item.name}
          </Link>
        ) : (
          <span className="text-gray-900 font-medium">{item.name}</span>
        )}
      </React.Fragment>
    ))}
  </nav>
);

// Page Header Component
interface PageHeaderProps {
  title: string;
  description?: string;
  breadcrumbs?: Breadcrumb[];
  children?: React.ReactNode;
}

export const PageHeader: React.FC<PageHeaderProps> = ({
  title,
  description,
  breadcrumbs,
  children
}) => (
  <div className="bg-gradient-to-b from-gray-50 to-white border-b border-gray-200">
    <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 py-8 sm:py-12">
      {breadcrumbs && <Breadcrumbs items={breadcrumbs} />}
      
      <div className="max-w-3xl">
        <h1 className="text-2xl sm:text-3xl lg:text-4xl font-black text-gray-900 mb-4">
          {title}
        </h1>
        {description && (
          <p className="text-lg sm:text-xl text-gray-600 leading-relaxed">
            {description}
          </p>
        )}
      </div>
      
      {children}
    </div>
  </div>
);

// Section Component
interface SectionProps {
  children: React.ReactNode;
  background?: 'white' | 'gray' | 'primary';
  padding?: 'sm' | 'md' | 'lg' | 'xl';
  className?: string;
  id?: string;
}

export const Section: React.FC<SectionProps> = ({
  children,
  background = 'white',
  padding = 'lg',
  className = '',
  id
}) => {
  const bgClasses = {
    white: 'bg-white',
    gray: 'bg-gray-100/50',
    primary: 'bg-primary-50'
  };

  const paddingClasses = {
    sm: 'py-8',
    md: 'py-12',
    lg: 'py-16',
    xl: 'py-18'
  };
  return (
    <section 
      className={`${bgClasses[background]} ${paddingClasses[padding]} ${className}`}
      id={id}
    >
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        {children}
      </div>
    </section>
  );
};

// Responsive Grid Component
interface ResponsiveGridProps {
  children: React.ReactNode;
  cols?: {
    default: 1 | 2 | 3 | 4 | 5 | 6;
    sm?: 1 | 2 | 3 | 4 | 5 | 6;
    md?: 1 | 2 | 3 | 4 | 5 | 6;
    lg?: 1 | 2 | 3 | 4 | 5 | 6;
    xl?: 1 | 2 | 3 | 4 | 5 | 6;
  };
  gap?: 'sm' | 'md' | 'lg' | 'xl';
  className?: string;
}

export const ResponsiveGrid: React.FC<ResponsiveGridProps> = ({
  children,
  cols = { default: 1 },
  gap = 'lg',
  className = ''
}) => {
  const colClasses: Record<1 | 2 | 3 | 4 | 5 | 6, string> = {
    1: 'grid-cols-1',
    2: 'grid-cols-2',
    3: 'grid-cols-3',
    4: 'grid-cols-4',
    5: 'grid-cols-5',
    6: 'grid-cols-6'
  };

  const gapClasses = {
    sm: 'gap-4',
    md: 'gap-6',
    lg: 'gap-8',
    xl: 'gap-12'
  };

  const gridCols = [
    colClasses[cols.default],
    cols.sm && `sm:${colClasses[cols.sm]}`,
    cols.md && `md:${colClasses[cols.md]}`,
    cols.lg && `lg:${colClasses[cols.lg]}`,
    cols.xl && `xl:${colClasses[cols.xl]}`
  ].filter(Boolean).join(' ');

  return (
    <div className={`grid ${gridCols} ${gapClasses[gap]} ${className}`}>
      {children}
    </div>
  );
};

// Main Page Layout Component
interface PageLayoutProps {
  children: React.ReactNode;
  title: string;
  description?: string;
  breadcrumbs?: Breadcrumb[];
  headerContent?: React.ReactNode;
}

export const PageLayout: React.FC<PageLayoutProps> = ({
  children,
  title,
  description,
  breadcrumbs,
  headerContent
}) => (
  <div className="min-h-screen bg-black">
    <PageHeader
      title={title}
      description={description}
      breadcrumbs={breadcrumbs}
    >
      {headerContent}
    </PageHeader>
    
    <Suspense
      fallback={
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          <div className="space-y-8">
            {[...Array(3)].map((_, i) => (
              <SkeletonLoader key={i} className="h-32 w-full" />
            ))}
          </div>
        </div>
      }
    >
      {children}
    </Suspense>
  </div>
);

export default PageLayout;
