'use client';

import { useState } from 'react';
import FileUpload from './file-upload';

interface ImageUploadFieldProps {
  label?: string;
  value?: string;
  onChange: (imageUrl: string) => void;
  onError?: (error: string) => void;
  oldImage?: string;
}

export function ImageUploadField({
  label = 'Profile Picture',
  value,
  onChange,
  onError,
  oldImage,
}: ImageUploadFieldProps) {
  const [file, setFile] = useState<File | null>(null);
  const [uploading, setUploading] = useState(false);
  const [error, setError] = useState<string>('');

  const handleFileChange = async (selectedFile: File | null) => {
    if (!selectedFile) {
      setFile(null);
      return;
    }

    setFile(selectedFile);
    setError('');
    setUploading(true);

    try {
      const formData = new FormData();
      formData.append('file', selectedFile);
      if (oldImage) {
        formData.append('oldImage', oldImage);
      }

      const response = await fetch('/api/admin/upload-profile-image', {
        method: 'POST',
        body: formData,
      });

      const data = await response.json();

      if (!response.ok) {
        const errorMsg = data.message || 'Failed to upload image';
        setError(errorMsg);
        onError?.(errorMsg);
        setFile(null);
      } else {
        onChange(data.data.url);
        setFile(null);
      }
    } catch (err) {
      const errorMsg = err instanceof Error ? err.message : 'Upload failed';
      setError(errorMsg);
      onError?.(errorMsg);
      setFile(null);
    } finally {
      setUploading(false);
    }
  };

  return (
    <div>
      <FileUpload
        label={label}
        accept="image/jpeg,image/png,image/webp"
        maxSize={5}
        value={file || value || null}
        onChange={handleFileChange}
        previewWidth={120}
        previewHeight={120}
        helpText="Max 5MB. Formats: JPEG, PNG, WebP"
        className={uploading ? 'opacity-50 cursor-not-allowed' : ''}
      />
      {error && <p className="mt-2 text-sm text-red-600">{error}</p>}
    </div>
  );
}

