"use client";

import React from "react";
import classNames from "classnames";
import SafeImage from "./safe-image";

interface AvatarProps {
  src?: string;
  name?: string;
  alt?: string;
  size?: "xs" | "sm" | "md" | "lg" | "xl" | "2xl";
  variant?: "circle" | "rounded" | "square";
  className?: string;
}

const Avatar: React.FC<AvatarProps> = ({
  src,
  name = "",
  alt,
  size = "md",
  variant = "circle",
  className = "",
}) => {
  // Generate initials from name
  const getInitials = (fullName: string): string => {
    const names = fullName.trim().split(" ");
    if (names.length === 1) {
      return names[0].charAt(0).toUpperCase();
    }
    return (names[0].charAt(0) + names[names.length - 1].charAt(0)).toUpperCase();
  };

  const initials = getInitials(name);

  // Size configurations
  const sizeConfig = {
    xs: {
      container: "w-6 h-6",
      text: "text-xs",
      image: { width: 24, height: 24 },
    },
    sm: {
      container: "w-8 h-8",
      text: "text-sm",
      image: { width: 32, height: 32 },
    },
    md: {
      container: "w-10 h-10",
      text: "text-base",
      image: { width: 40, height: 40 },
    },
    lg: {
      container: "w-12 h-12",
      text: "text-lg",
      image: { width: 48, height: 48 },
    },
    xl: {
      container: "w-16 h-16",
      text: "text-xl",
      image: { width: 64, height: 64 },
    },
    "2xl": {
      container: "w-20 h-20",
      text: "text-2xl",
      image: { width: 80, height: 80 },
    },
  };

  // Variant configurations
  const variantClass = {
    circle: "rounded-full",
    rounded: "rounded-lg",
    square: "rounded-none",
  };

  const currentSize = sizeConfig[size];

  // Base classes
  const baseClass = classNames(
    "inline-flex items-center justify-center overflow-hidden",
    "bg-gradient-to-br from-primary-500 to-primary-600",
    "text-white font-semibold",
    "border-2 border-primary shadow-sm",
    currentSize.container,
    variantClass[variant],
    className
  );

  // If no image source, show initials
  if (!src) {
    return (
      <div className={baseClass} role="img" aria-label={`Avatar for ${name || "user"}`}>
        <span className={classNames("leading-none", currentSize.text)}>
          {initials}
        </span>
      </div>
    );
  }

  // If image source exists, show image with fallback to initials
  return (
    <div className={baseClass}>
      <SafeImage
        src={src}
        alt={alt || `${name}'s avatar`}
        width={currentSize.image.width}
        height={currentSize.image.height}
        className="w-full h-full object-cover"
        fallbackSvg={`data:image/svg+xml;base64,${btoa(
          `<svg width="${currentSize.image.width}" height="${currentSize.image.height}" xmlns="http://www.w3.org/2000/svg">
            <rect width="${currentSize.image.width}" height="${currentSize.image.height}" fill="#6366f1"/>
            <text x="50%" y="50%" font-family="system-ui" font-size="${
              size === "xs" ? "8" : size === "sm" ? "10" : size === "md" ? "12" : size === "lg" ? "14" : size === "xl" ? "16" : "18"
            }" fill="white" text-anchor="middle" dy=".3em">${initials}</text>
          </svg>`
        )}`}
      />
    </div>
  );
};

export default Avatar;