import React from "react";
import classNames from "classnames";
import { ArrowPathIcon } from "@heroicons/react/24/outline";

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: "contained" | "outlined" | "text";
  color?: 
    | "primary" 
    | "secondary" 
    | "accent" 
    | "success" 
    | "error" 
    | "danger" 
    | "dark";
  size?: "sm" | "md" | "lg";
  fullWidth?: boolean;
  loading?: boolean;
}

const Button: React.FC<ButtonProps> = ({
  children,
  variant = "contained",
  color = "primary",
  size = "md",
  fullWidth = false,
  loading = false,
  className = "",
  ...props
}) => {
  const baseClass =
    "rounded-lg font-medium shadow-sm transition-all duration-200 cursor-pointer inline-flex items-center justify-center gap-2 disabled:cursor-not-allowed";

  const sizeClass = {
    sm: "px-3 py-1.5 text-sm",
    md: "px-4 py-2 text-base",
    lg: "px-5 py-3 text-lg",
  };

  const colorClass = {
    // THEME COLORS (Primary & Secondary)
    primary: {
      contained:
        "bg-gradient-to-r from-primary-600 to-primary-700 text-white hover:from-primary-700 hover:to-primary-800 active:from-primary-800 active:to-primary-900 disabled:from-primary-300 disabled:to-primary-400 shadow-lg hover:shadow-xl",
      outlined:
        "border-2 border-primary-600! text-primary-600 hover:bg-primary-50 active:bg-primary-100 disabled:border-primary-300! disabled:text-primary-300 shadow-xs!",
      text: "text-primary-600 hover:bg-primary-50 active:bg-primary-100 disabled:text-primary-300 shadow-none!",
    },
    secondary: {
      contained:
        "bg-gradient-to-r from-purple-600 to-purple-700 text-white hover:from-purple-700 hover:to-purple-800 active:from-purple-800 active:to-purple-900 disabled:from-purple-300 disabled:to-purple-400 shadow-lg hover:shadow-xl",
      outlined:
        "border-2 border-purple-600! text-purple-600 hover:bg-purple-50 active:bg-purple-100 disabled:border-purple-300! disabled:text-purple-300",
      text: "text-purple-600 hover:bg-purple-50 active:bg-purple-100 disabled:text-purple-300 !shadow-none",
    },
    // ACCENT COLORS (Additional Options)
    accent: {
      contained:
        "bg-indigo-600 text-white hover:bg-indigo-700 active:bg-indigo-800 disabled:bg-indigo-300",
      outlined:
        "border-2 border-indigo-600! text-indigo-600 hover:bg-indigo-50 active:bg-indigo-100 disabled:border-indigo-300! disabled:text-indigo-300",
      text: "text-indigo-600 hover:bg-indigo-50 active:bg-indigo-100 disabled:text-indigo-300 !shadow-none",
    },
    success: {
      contained:
        "bg-green-600 text-white hover:bg-green-700 active:bg-green-800 disabled:bg-green-300",
      outlined:
        "border-2 border-green-600! text-green-600 hover:bg-green-50 active:bg-green-100 disabled:border-green-300! disabled:text-green-300",
      text: "text-green-600 hover:bg-green-50 active:bg-green-100 disabled:text-green-300 !shadow-none",
    },
    error: {
      contained:
        "bg-red-600 text-white hover:bg-red-700 active:bg-red-800 disabled:bg-red-300",
      outlined:
        "border-2 border-red-600 text-red-600 hover:bg-red-50 active:bg-red-100 disabled:border-red-300 disabled:text-red-300",
      text: "text-red-600 hover:bg-red-50 active:bg-red-100 disabled:text-red-300 !shadow-none",
    },
    danger: {
      contained:
        "bg-red-700 text-white hover:bg-red-800 active:bg-red-900 disabled:bg-red-400",
      outlined:
        "border-2 border-red-700! text-red-700 hover:bg-red-50 active:bg-red-100 disabled:border-red-400! disabled:text-red-400",
      text: "text-red-700 hover:bg-red-50 active:bg-red-100 disabled:text-red-400 !shadow-none",
    },
    dark: {
      contained:
        "bg-neutral-700 text-white hover:bg-neutral-800 active:bg-gray-950 disabled:bg-gray-500",
      outlined:
        "border-2 border-neutral-700! text-neutral-700 hover:bg-gray-50 active:bg-gray-100 disabled:border-gray-500! disabled:text-gray-500",
      text: "text-gray-700 hover:bg-neutral-50 active:bg-neutral-100 disabled:text-neutral-500 !shadow-none",
    },
  };

  const iconSize = {
    sm: "w-4 h-4",
    md: "w-5 h-5",
    lg: "w-6 h-6",
  };

  return (
    <button
      className={classNames(
        baseClass,
        sizeClass[size],
        colorClass[color][variant],
        fullWidth && "w-full",
        props.disabled && "opacity-50",
        className
      )}
      {...props}
    >
      {loading && (
        <ArrowPathIcon
          className={classNames("animate-spin", iconSize[size])}
        />
      )}
      {children}
    </button>
  );
};

export default Button;



