"use client";

import { useState, useEffect } from "react";
import Image from "next/image";
import { Upload, X } from "lucide-react";

interface FileUploadProps {
  label?: string;
  accept?: string;
  maxSize?: number; // in MB
  value?: File | null | string;
  onChange: (file: File | null) => void;
  previewWidth?: number;
  previewHeight?: number;
  helpText?: string;
  className?: string;
}

export default function FileUpload({
  label = "Upload File",
  accept = "image/jpeg,image/jpg,image/png,image/webp",
  maxSize = 2,
  value = null,
  onChange,
  previewWidth = 300,
  previewHeight = 200,
  helpText,
  className = "",
}: FileUploadProps) {
  const [error, setError] = useState<string>("");
  const [preview, setPreview] = useState<string | null>(null);

  // Update preview when value prop changes (from string URL or File object)
  useEffect(() => {
    if (!value) {
      setPreview(null);
      return;
    }

    if (typeof value === "string") {
      // Value is a URL string (existing image when editing)
      setPreview(value);
    } else if (value instanceof File) {
      // Value is a File object - create preview
      const reader = new FileReader();
      reader.onloadend = () => {
        setPreview(reader.result as string);
      };
      reader.readAsDataURL(value);
    }
  }, [value]);

  const validateFile = (file: File): boolean => {
    setError("");

    // Check file type
    const acceptedTypes = accept.split(",").map((type) => type.trim());
    if (!acceptedTypes.includes(file.type)) {
      setError(
        `Invalid file type. Please upload: ${acceptedTypes
          .map((t) => t.split("/")[1].toUpperCase())
          .join(", ")}`
      );
      return false;
    }

    // Check file size
    const maxSizeBytes = maxSize * 1024 * 1024;
    if (file.size > maxSizeBytes) {
      setError(`File size must be less than ${maxSize}MB`);
      return false;
    }

    return true;
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    if (!validateFile(file)) {
      e.target.value = "";
      return;
    }

    try {
      // Store the File object
      onChange(file);
      setError("");

      // Create preview
      const reader = new FileReader();
      reader.onloadend = () => {
        setPreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    } catch {
      setError("Failed to read file");
      e.target.value = "";
    }
  };

  const handleRemove = () => {
    onChange(null);
    setPreview(null);
    setError("");
  };

  const getAcceptedFormats = () => {
    return accept
      .split(",")
      .map((type) => type.split("/")[1].toUpperCase())
      .join(", ");
  };

  return (
    <div className={className}>
      {label && (
        <label className="block text-sm font-medium text-gray-700 mb-2">
          {label}
        </label>
      )}
      {!preview ? (
        <div>
          <label
            htmlFor="file-upload"
            className="block border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-blue-500 transition cursor-pointer"
          >
            <Upload className="mx-auto h-12 w-12 text-gray-400" />
            <p className="mt-2 text-sm text-gray-600">
              Click to upload or drag and drop
            </p>
            <p className="text-xs text-gray-500 mt-1">
              {helpText ||
                `${getAcceptedFormats()} (max ${maxSize}MB)`}
            </p>
            <input
              id="file-upload"
              type="file"
              className="hidden"
              accept={accept}
              onChange={handleFileChange}
            />
          </label>
          {error && (
            <p className="mt-2 text-sm text-red-600">{error}</p>
          )}
        </div>
      ) : (
        <div className="relative inline-block">
          <div
            className="relative rounded-lg overflow-hidden border border-gray-200"
            style={{
              width: previewWidth,
              height: previewHeight,
            }}
          >
            <Image
              src={preview || ""}
              alt="Preview"
              fill
              className="object-cover"
            />
          </div>
          <button
            type="button"
            onClick={handleRemove}
            className="absolute -top-2 -right-2 p-1 bg-red-600 text-white rounded-full hover:bg-red-700 transition"
          >
            <X className="w-4 h-4" />
          </button>
        </div>
      )}
    </div>
  );
}
