"use client";

import { InputHTMLAttributes, useState } from "react";
import { Eye, EyeOff } from "lucide-react";
import classNames from "classnames";

interface InputFieldProps extends InputHTMLAttributes<HTMLInputElement> {
  label?: string | React.ReactNode;
  error?: string;
  helperText?: string;
  showPasswordToggle?: boolean;
  icon?: React.ReactNode;
  iconPosition?: "end" | "start";
  inputSize?: "sm" | "md" | "lg";
  containerClassName?: string;
  labelClassName?: string;
  inputClassName?: string;
  errorClassName?: string;
  helperClassName?: string;
}

export default function InputField({
  label,
  error,
  helperText,
  showPasswordToggle = false,
  icon,
  iconPosition = "end",
  inputSize = "md",
  type = "text",
  containerClassName,
  labelClassName,
  inputClassName,
  errorClassName,
  helperClassName,
  className,
  id,
  ...props
}: InputFieldProps) {
  const [showPassword, setShowPassword] = useState(false);

  const isPasswordField = type === "password" && showPasswordToggle;
  const inputType = isPasswordField && showPassword ? "text" : type;

  // Size classes
  const sizeClasses = {
    sm: "px-3 py-1 text-sm",
    md: "px-4 py-2 text-base",
    lg: "px-5 py-3 text-lg",
  };

  // Default styles
  const defaultContainerClassName = "w-full flex flex-col space-y-1";
  const defaultLabelClassName = "text-sm font-medium text-gray-700 cursor-pointer";
  const defaultInputClassName =
    "w-full rounded-lg bg-white border border-gray-300 outline-1 outline-gray-300 ring-gray-300 disabled:border-gray-200 focus:outline-2 focus:ring-2 focus:ring-blue-500 focus:outline-blue-500 transition";
  const defaultErrorClassName = "text-sm text-red-600";
  const defaultHelperClassName = "text-xs text-gray-500";

  // Determine padding for icons
  const hasIcon = icon !== undefined;
  const paddingClass = classNames(
    hasIcon && iconPosition === "end" && "pr-12",
    hasIcon && iconPosition === "start" && "pl-12"
  );

  const finalInputClassName = classNames(
    defaultInputClassName,
    sizeClasses[inputSize],
    {
      "border-red-500 focus:ring-red-500 focus:outline-red-500": error,
      "pr-12": isPasswordField || (hasIcon && iconPosition === "end"),
    },
    paddingClass,
    inputClassName
  );

  return (
    <div className={containerClassName || defaultContainerClassName}>
      {label && (
        <label
          htmlFor={id}
          className={labelClassName || defaultLabelClassName}
        >
          {label}
        </label>
      )}

      <div className="relative">
        <input
          id={id}
          type={inputType}
          className={finalInputClassName}
          {...props}
        />

        {/* Password Toggle Icon */}
        {isPasswordField && (
          <button
            type="button"
            onClick={() => setShowPassword(!showPassword)}
            className="absolute right-0 top-1/2 -translate-y-1/2 w-10 h-full flex items-center justify-center text-gray-600 hover:text-gray-800 hover:bg-gray-50 transition-colors rounded-r-lg"
            tabIndex={-1}
            aria-label={showPassword ? "Hide password" : "Show password"}
          >
            {showPassword ? (
              <EyeOff className="w-5 h-5" />
            ) : (
              <Eye className="w-5 h-5" />
            )}
          </button>
        )}

        {/* Custom Icon */}
        {icon && !isPasswordField && (
          <div
            className={classNames(
              "absolute inset-y-0 w-10 hover:bg-gray-50 transition-colors",
              iconPosition === "end" && "right-0 rounded-r-lg",
              iconPosition === "start" && "left-0 rounded-l-lg"
            )}
          >
            <span
              className={classNames(
                "w-fit absolute top-1/2 -translate-y-1/2 text-gray-600",
                iconPosition === "end" && "right-3",
                iconPosition === "start" && "left-3"
              )}
            >
              {icon}
            </span>
          </div>
        )}
      </div>

      {/* Error Message */}
      {error && (
        <p className={errorClassName || defaultErrorClassName}>{error}</p>
      )}

      {/* Helper Text */}
      {helperText && !error && (
        <p className={helperClassName || defaultHelperClassName}>{helperText}</p>
      )}
    </div>
  );
}
