"use client";

import { Dialog, DialogPanel, DialogTitle } from "@headlessui/react";
import classNames from "classnames";
import { LuX } from "react-icons/lu";

interface ModalProps {
  isOpen: boolean;
  onClose: () => void;
  children: React.ReactNode;
  title?: string;
  size?: "sm" | "md" | "lg" | "xl";
  placement?: "center" | "right" | "left" | "top" | "bottom";
  className?: string;
  onOutSide?: boolean;
}

export default function Modal({
  isOpen,
  onClose,
  children,
  title,
  size = "md",
  placement = "center",
  className,
  onOutSide,
}: ModalProps) {
  const sizeClasses = {
    sm: "max-w-[300px]",
    md: "max-w-[500px]",
    lg: "max-w-[700px]",
    xl: "max-w-[900px]",
  };

  const placementClasses = {
    center: "flex items-center justify-center inset-0 p-5",
    right: "fixed inset-0 flex justify-end pl-5",
    left: "fixed inset-0 flex justify-start pr-5",
    top: "fixed inset-0 flex items-start justify-center pb-20",
    bottom: "fixed inset-0 flex items-end justify-center pt-20",
  };

  const placementInnerContainer = {
    left: "h-full rounded-l-none data-[enter]:duration-150 data-[leave]:duration-300 data-[closed]:-translate-x-full data-[closed]:opacity-0",
    right:
      "h-full rounded-r-none data-[enter]:duration-150 data-[leave]:duration-300 data-[closed]:translate-x-full data-[closed]:opacity-0",
    top: "rounded-t-none mx-auto max-w-6xl w-full data-[enter]:duration-150 data-[leave]:duration-300 data-[closed]:-translate-y-full data-[closed]:opacity-0",
    bottom:
      "rounded-b-none mx-auto max-w-6xl w-full data-[enter]:duration-150 data-[leave]:duration-300 data-[closed]:translate-y-full data-[closed]:opacity-0",
    center:
      "mx-auto rounded-xl max-w-6xl w-full data-[enter]:duration-150 data-[leave]:duration-300 data-[closed]:scale-95 data-[closed]:opacity-0",
  };

  return (
    <Dialog
      open={isOpen}
      transition
      as="div"
      className="relative z-[999] focus:outline-none h-screen"
      onClose={() => (onOutSide ? null : onClose())}
    >
      {/* Overlay */}
      <div className="fixed inset-0 bg-black/50 backdrop-blur-sm" />

      {/* Modal Container */}
      <div className={classNames("fixed", placementClasses[placement])}>
        <DialogPanel
          transition
          className={classNames(
            "bg-white rounded-xl px-5 pb-6 shadow-lg transition-all duration-300 ease-out w-full relative overflow-hidden",
            sizeClasses[size],
            placementInnerContainer[placement],
            className
          )}
        >
          {/* Header */}
          {title ? (
            <div
              className={classNames(
                "flex pt-6 justify-between items-center border-b border-gray-200 pb-3 sticky top-0 inset-x-0 bg-white z-10",
                !title && "border-none"
              )}
            >
              <DialogTitle className="text-lg font-semibold">
                <p>{title}</p>
              </DialogTitle>
              <button
                className="p-2 text-gray-500 hover:text-gray-700"
                onClick={onClose}
              >
                <LuX className="w-5 h-5" />
              </button>
            </div>
          ) : (
            <button
              className="p-2 absolute top-2 right-2 text-gray-500 hover:text-gray-700"
              onClick={onClose}
            >
              <LuX className="w-5 h-5" />
            </button>
          )}

          {/* Body (auto height, scrolls only if overflow) */}
          <div className="mt-4 px-0.5 w-full max-h-[80vh] overflow-y-auto">
            {children}
          </div>
        </DialogPanel>
      </div>
    </Dialog>
  );
}