"use client";

import React, { useEffect } from "react";
import { useEditor, useEditorState, EditorContent } from "@tiptap/react";
import StarterKit from "@tiptap/starter-kit";
import Underline from "@tiptap/extension-underline";
import Placeholder from "@tiptap/extension-placeholder";
import ResizableImageExtension from "tiptap-extension-resize-image";
import Link from "@tiptap/extension-link";
import { TextAlign } from "@tiptap/extension-text-align";
import {
  Bold,
  Italic,
  Underline as UnderlineIcon,
  Heading1,
  Heading2,
  List,
  ListOrdered,
  Link as LinkIcon,
  Image as ImageIcon,
  Undo2,
  Redo2,
  AlignLeft,
  AlignCenter,
  AlignRight,
  AlignJustify,
  Pilcrow,
} from "lucide-react";

interface RichTextEditorProps {
  value: string;
  onChange: (value: string) => void;
  readOnly?: boolean;
}

export default function RichTextEditor({
  value,
  onChange,
  readOnly = false,
}: RichTextEditorProps) {
  const editor = useEditor({
    extensions: [
      StarterKit,
      Underline,
      ResizableImageExtension.configure({
        inline: true,
        allowBase64: true,
      }),
      Link.configure({
        openOnClick: false,
        HTMLAttributes: {
          class: "text-blue-600 underline cursor-pointer hover:text-blue-800",
        },
      }),
      TextAlign.configure({
        types: ["heading", "paragraph"],
      }),
      Placeholder.configure({
        placeholder: "Start writing your content...",
      }),
    ],
    content: value,
    onUpdate: ({ editor }) => {
      onChange(editor.getHTML());
    },
    editable: !readOnly,
    immediatelyRender: false,
  });

  // Update editor content when value prop changes
  useEffect(() => {
    if (editor && value && editor.getHTML() !== value) {
      editor.commands.setContent(value);
    }
  }, [value, editor]);

  if (!editor) {
    return (
      <div className="bg-gray-100 rounded-lg p-4 min-h-[400px]">
        Loading editor...
      </div>
    );
  }

  if (readOnly) {
    return (
      <div className="prose prose-sm max-w-none bg-gray-50 rounded-lg p-4 border border-gray-200">
        <EditorContent editor={editor} />
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg border border-gray-300 overflow-hidden">
      <Toolbar editor={editor} />
      <div className="prose prose-sm max-w-none p-4 min-h-[400px]">
        <style jsx global>{`
          /* TipTap Placeholder Styles */
          .tiptap p.is-editor-empty:first-child::before {
            color: #adb5bd;
            content: attr(data-placeholder);
            float: left;
            height: 0;
            pointer-events: none;
          }

          /* TipTap Editor Focus Styles */
          .tiptap:focus {
            outline: none;
          }

          /* TipTap Content Styles */
          .tiptap {
            min-height: 350px;
          }

          .tiptap > * + * {
            margin-top: 0.75em;
          }

          .tiptap ul,
          .tiptap ol {
            padding: 0 1rem;
            margin: 0.75em 0;
            list-style-position: outside;
          }

          .tiptap ul {
            list-style-type: disc;
          }

          .tiptap ol {
            list-style-type: decimal;
          }

          .tiptap ul ul {
            list-style-type: circle;
          }

          .tiptap li {
            margin-left: 1em;
          }

          .tiptap li p {
            margin: 0;
          }

          .tiptap h1,
          .tiptap h2,
          .tiptap h3,
          .tiptap h4,
          .tiptap h5,
          .tiptap h6 {
            line-height: 1.1;
            margin-top: 1.5em;
            margin-bottom: 0.5em;
          }

          .tiptap h1 {
            font-size: 2em;
            font-weight: bold;
          }

          .tiptap h2 {
            font-size: 1.5em;
            font-weight: bold;
          }

          .tiptap code {
            background-color: rgba(97, 97, 97, 0.1);
            color: #616161;
            padding: 0.2em 0.4em;
            border-radius: 0.25em;
            font-size: 0.85em;
          }

          .tiptap pre {
            background: #0d0d0d;
            color: #fff;
            font-family: "JetBrainsMono", monospace;
            padding: 0.75rem 1rem;
            border-radius: 0.5rem;
            overflow-x: auto;
          }

          .tiptap img {
            max-width: 100%;
            height: auto;
            display: block;
            margin: 1em 0;
          }

          .tiptap img.ProseMirror-selectednode {
            outline: 3px solid #3b82f6;
            cursor: nwse-resize;
          }

          .tiptap blockquote {
            padding-left: 1rem;
            border-left: 3px solid rgba(13, 13, 13, 0.1);
          }

          .tiptap hr {
            border: none;
            border-top: 2px solid rgba(13, 13, 13, 0.1);
            margin: 2rem 0;
          }

          /* Text Alignment Styles */
          .tiptap [style*="text-align: left"],
          .tiptap p[style*="text-align: left"],
          .tiptap h1[style*="text-align: left"],
          .tiptap h2[style*="text-align: left"] {
            text-align: left !important;
          }

          .tiptap [style*="text-align: center"],
          .tiptap p[style*="text-align: center"],
          .tiptap h1[style*="text-align: center"],
          .tiptap h2[style*="text-align: center"] {
            text-align: center !important;
          }

          .tiptap [style*="text-align: right"],
          .tiptap p[style*="text-align: right"],
          .tiptap h1[style*="text-align: right"],
          .tiptap h2[style*="text-align: right"] {
            text-align: right !important;
          }

          .tiptap [style*="text-align: justify"],
          .tiptap p[style*="text-align: justify"],
          .tiptap h1[style*="text-align: justify"],
          .tiptap h2[style*="text-align: justify"] {
            text-align: justify !important;
          }
        `}</style>
        <EditorContent editor={editor} />
      </div>
    </div>
  );
}

interface ToolbarProps {
  editor: ReturnType<typeof useEditor>;
}

function Toolbar({ editor }: ToolbarProps) {
  const editorState = useEditorState({
    editor,
    selector: (ctx) => ({
      isBold: ctx.editor.isActive("bold"),
      isItalic: ctx.editor.isActive("italic"),
      isUnderline: ctx.editor.isActive("underline"),
      isHeading1: ctx.editor.isActive("heading", { level: 1 }),
      isHeading2: ctx.editor.isActive("heading", { level: 2 }),
      isParagraph: ctx.editor.isActive("paragraph"),
      isBulletList: ctx.editor.isActive("bulletList"),
      isOrderedList: ctx.editor.isActive("orderedList"),
      isAlignLeft: ctx.editor.isActive({ textAlign: "left" }),
      isAlignCenter: ctx.editor.isActive({ textAlign: "center" }),
      isAlignRight: ctx.editor.isActive({ textAlign: "right" }),
      isAlignJustify: ctx.editor.isActive({ textAlign: "justify" }),
      canUndo: ctx.editor.can().chain().undo().run(),
      canRedo: ctx.editor.can().chain().redo().run(),
    }),
  });

  const handleImageUpload = () => {
    const input = document.createElement("input");
    input.type = "file";
    input.accept = "image/*";
    
    input.onchange = async (e) => {
      const file = (e.target as HTMLInputElement).files?.[0];
      if (!file) return;

      // Validation: Check file type
      const validTypes = ["image/jpeg", "image/jpg", "image/png", "image/gif", "image/webp"];
      if (!validTypes.includes(file.type)) {
        alert("Please upload a valid image file (JPEG, PNG, GIF, or WebP)");
        return;
      }

      // Validation: Check file size (max 5MB)
      const maxSize = 5 * 1024 * 1024; // 5MB in bytes
      if (file.size > maxSize) {
        alert("Image size must be less than 5MB. Please choose a smaller image.");
        return;
      }

      const reader = new FileReader();
      reader.onload = (event) => {
        const url = event.target?.result as string;
        
        // Set image with default width of 600px (user can resize after)
        editor
          .chain()
          .focus()
          .setImage({ 
            src: url,
            alt: file.name.replace(/\.[^/.]+$/, ""), // filename without extension
            width: 500
          })
          .run();
      };
      reader.readAsDataURL(file);
    };
    input.click();
  };

  const handleLinkAdd = () => {
    const previousUrl = editor.getAttributes("link").href;
    const url = prompt("Enter URL:", previousUrl);

    // cancelled
    if (url === null) {
      return;
    }

    // empty
    if (url === "") {
      editor.chain().focus().extendMarkRange("link").unsetLink().run();
      return;
    }

    // update link
    editor.chain().focus().extendMarkRange("link").setLink({ href: url }).run();
  };

  return (
    <div className="flex flex-wrap gap-1 border-b border-gray-300 bg-gray-50 p-3">
      <ToolbarButton
        onClick={() => editor.chain().focus().toggleBold().run()}
        active={editorState.isBold}
        title="Bold (Ctrl+B)"
        icon={<Bold size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().toggleItalic().run()}
        active={editorState.isItalic}
        title="Italic (Ctrl+I)"
        icon={<Italic size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().toggleUnderline().run()}
        active={editorState.isUnderline}
        title="Underline (Ctrl+U)"
        icon={<UnderlineIcon size={18} />}
      />

      <div className="w-px bg-gray-300" />

      <ToolbarButton
        onClick={() => editor.chain().focus().toggleHeading({ level: 1 }).run()}
        active={editorState.isHeading1}
        title="Heading 1"
        icon={<Heading1 size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().toggleHeading({ level: 2 }).run()}
        active={editorState.isHeading2}
        title="Heading 2"
        icon={<Heading2 size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().setParagraph().run()}
        active={editorState.isParagraph}
        title="Paragraph"
        icon={<Pilcrow size={18} />}
      />

      <div className="w-px bg-gray-300" />

      <ToolbarButton
        onClick={() => editor.chain().focus().unsetTextAlign().run()}
        active={editorState.isAlignLeft}
        title="Align Left"
        icon={<AlignLeft size={18} />}
      />
      <ToolbarButton
        onClick={() =>
          editor.chain().focus().setTextAlign("center").run()
        }
        active={editorState.isAlignCenter}
        title="Align Center"
        icon={<AlignCenter size={18} />}
      />
      <ToolbarButton
        onClick={() =>
          editor.chain().focus().setTextAlign("right").run()
        }
        active={editorState.isAlignRight}
        title="Align Right"
        icon={<AlignRight size={18} />}
      />
      <ToolbarButton
        onClick={() =>
          editor.chain().focus().setTextAlign("justify").run()
        }
        active={editorState.isAlignJustify}
        title="Align Justify"
        icon={<AlignJustify size={18} />}
      />

      <div className="w-px bg-gray-300" />

      <ToolbarButton
        onClick={() => editor.chain().focus().toggleBulletList().run()}
        active={editorState.isBulletList}
        title="Bullet List"
        icon={<List size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().toggleOrderedList().run()}
        active={editorState.isOrderedList}
        title="Numbered List"
        icon={<ListOrdered size={18} />}
      />

      <div className="w-px bg-gray-300" />

      <ToolbarButton
        onClick={handleLinkAdd}
        active={editor.isActive("link")}
        title="Add Link"
        icon={<LinkIcon size={18} />}
      />
      <ToolbarButton
        onClick={handleImageUpload}
        title="Add Image"
        icon={<ImageIcon size={18} />}
      />

      <div className="w-px bg-gray-300" />

      <ToolbarButton
        onClick={() => editor.chain().focus().undo().run()}
        disabled={!editorState.canUndo}
        title="Undo"
        icon={<Undo2 size={18} />}
      />
      <ToolbarButton
        onClick={() => editor.chain().focus().redo().run()}
        disabled={!editorState.canRedo}
        title="Redo"
        icon={<Redo2 size={18} />}
      />
    </div>
  );
}

interface ToolbarButtonProps {
  onClick: () => void;
  active?: boolean;
  disabled?: boolean;
  title: string;
  icon: React.ReactNode;
}

function ToolbarButton({
  onClick,
  active = false,
  disabled = false,
  title,
  icon,
}: ToolbarButtonProps) {
  return (
    <button
      onClick={onClick}
      disabled={disabled}
      title={title}
      className={`p-2 rounded hover:bg-gray-200 transition-colors flex items-center justify-center ${
        active ? "bg-blue-100 text-blue-600" : "text-gray-700"
      } ${disabled ? "opacity-50 cursor-not-allowed" : "cursor-pointer"}`}
    >
      {icon}
    </button>
  );
}
