"use client";

import { useState, useRef, useEffect } from "react";
import { createPortal } from "react-dom";
import classNames from "classnames";
import { LuCheck, LuChevronDown, LuX } from "react-icons/lu";

interface DropdownOption {
  label: string;
  value: string | number;
  subtitle?: string;
  disabled?: boolean;
}

interface SearchableDropdownProps {
  options: DropdownOption[];
  value: string | number | (string | number)[] | null;
  placeholder?: string;
  label?: string;
  required?: boolean;
  disabled?: boolean;
  className?: string;
  inputSize?: "sm" | "md" | "lg";
  multiple?: boolean;
  onChange: (value: string | number | (string | number)[] | null) => void;
  onSearch?: (searchTerm: string) => void;
  loading?: boolean;
  clearable?: boolean;
  maxHeight?: string;
  noResultsText?: string;
  error?: string;
  limitTags?: number;
}

export default function SearchableDropdown({
  options,
  value,
  placeholder = "Select an option...",
  label,
  required = false,
  disabled = false,
  className = "",
  inputSize = "md",
  multiple = false,
  onChange,
  onSearch,
  loading = false,
  clearable = true,
  maxHeight = "200px",
  noResultsText = "No results found",
  error,
  limitTags = 0,
}: SearchableDropdownProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [highlightedIndex, setHighlightedIndex] = useState(-1);
  const [mounted, setMounted] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState<{top: number; left: number; width: number; isFlipped: boolean} | null>(null);

  const dropdownRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);
  const optionsRef = useRef<HTMLDivElement>(null);
  const portalRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    setMounted(true);
  }, []);

  useEffect(() => {
    const updatePosition = () => {
      if (isOpen && dropdownRef.current) {
        const getDropdownHeight = (): number => {
          const match = maxHeight.match(/(\d+)/);
          return match ? parseInt(match[0], 10) : 200;
        };

        const rect = dropdownRef.current.getBoundingClientRect();
        const dropdownHeight = getDropdownHeight();
        const spaceBelow = window.innerHeight - rect.bottom;
        const spaceAbove = rect.top;
        
        // Flip if not enough space below AND more space above
        const shouldFlip = spaceBelow < dropdownHeight && spaceAbove > spaceBelow;
        
        setDropdownPosition({
          top: shouldFlip 
            ? rect.top + window.scrollY - dropdownHeight 
            : rect.bottom + window.scrollY,
          left: rect.left,
          width: rect.width,
          isFlipped: shouldFlip,
        });
      }
    };

    if (isOpen) {
      window.addEventListener("scroll", updatePosition, true); // true = capture phase
      window.addEventListener("resize", updatePosition);
      updatePosition();
    }

    return () => {
      window.removeEventListener("scroll", updatePosition, true);
      window.removeEventListener("resize", updatePosition);
    };
  }, [isOpen, maxHeight]);

  const sizeClasses = {
    sm: "px-3 py-1 text-sm",
    md: "px-4 py-2 text-base",
    lg: "px-5 py-3 text-lg",
  };

  const selectedOptions = multiple
    ? options.filter((option) =>
        Array.isArray(value) ? value.includes(option.value) : false
      )
    : options.find((option) => option.value === value);

  const filteredOptions = options?.filter(
    (option) =>
      option?.label?.toLowerCase()?.includes(searchTerm.toLowerCase()) ||
      (option?.subtitle &&
        option?.subtitle?.toLowerCase()?.includes(searchTerm.toLowerCase()))
  );

  useEffect(() => {
    if (onSearch) {
      const debounceTimer = setTimeout(() => {
        onSearch(searchTerm);
      }, 300);
      return () => clearTimeout(debounceTimer);
    }
  }, [searchTerm, onSearch]);

  const isOptionSelected = (option: DropdownOption) => {
    if (multiple) {
      return Array.isArray(value) ? value.includes(option.value) : false;
    }
    return option.value === value;
  };

  const hasAnySelection = () => {
    if (multiple) {
      return Array.isArray(value) && value.length > 0;
    }
    return !!value;
  };

  const clearSelection = (e: React.MouseEvent) => {
    e.stopPropagation();
    onChange(multiple ? [] : null);
    setSearchTerm("");
  };

  const removeSelectedOption = (optionToRemove: DropdownOption, e: React.MouseEvent) => {
    e.stopPropagation();
    if (multiple && Array.isArray(value)) {
      const newValues = value.filter((v) => v !== optionToRemove.value);
      onChange(newValues);
    }
  };

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      // Check if click is outside both the dropdown trigger AND the portal dropdown
      const isClickOutsideDropdown = dropdownRef.current && !dropdownRef.current.contains(event.target as Node);
      const isClickOutsidePortal = portalRef.current && !portalRef.current.contains(event.target as Node);
      
      if (isClickOutsideDropdown && isClickOutsidePortal) {
        setIsOpen(false);
        setSearchTerm("");
        setHighlightedIndex(-1);
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
      return () => document.removeEventListener("mousedown", handleClickOutside);
    }
  }, [isOpen]);

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (!isOpen) {
      if (["Enter", "ArrowDown", "ArrowUp"].includes(e.key)) {
        e.preventDefault();
        setIsOpen(true);
      }
      return;
    }

    switch (e.key) {
      case "ArrowDown":
        e.preventDefault();
        setHighlightedIndex((prev) =>
          prev < filteredOptions.length - 1 ? prev + 1 : 0
        );
        break;
      case "ArrowUp":
        e.preventDefault();
        setHighlightedIndex((prev) =>
          prev > 0 ? prev - 1 : filteredOptions.length - 1
        );
        break;
      case "Enter":
        e.preventDefault();
        if (highlightedIndex >= 0) {
          const option = filteredOptions[highlightedIndex];
          if (!option.disabled) handleOptionClick(option);
        }
        break;
      case "Escape":
        setIsOpen(false);
        setSearchTerm("");
        setHighlightedIndex(-1);
        break;
    }
  };

  useEffect(() => {
    if (highlightedIndex >= 0 && optionsRef.current) {
      const highlightedElement = optionsRef.current.children[highlightedIndex];
      if (highlightedElement) {
        (highlightedElement as HTMLElement).scrollIntoView({
          block: "nearest",
          behavior: "smooth",
        });
      }
    }
  }, [highlightedIndex]);

  const handleOptionClick = (option: DropdownOption) => {
    if (option.disabled) return;

    if (multiple) {
      if (Array.isArray(value)) {
        const isSelected = value.includes(option.value);
        const newValues = isSelected
          ? value.filter((v) => v !== option.value)
          : [...value, option.value];
        onChange(newValues);
      } else {
        onChange([option.value]);
      }
      setSearchTerm("");
    } else {
      onChange(option.value);
      setIsOpen(false);
      setSearchTerm("");
      setHighlightedIndex(-1);
    }
  };

  return (
    <div
      className={classNames("relative space-y-1", className)}
      ref={dropdownRef}
    >
      {label && (
        <label
          htmlFor={inputRef.current?.id}
          className="block text-sm font-semibold text-gray-800"
        >
          {label}
          {required && <span className="text-red-600 ml-1">*</span>}
        </label>
      )}

      <div
        className={classNames(
          "relative w-full rounded-lg bg-white border border-gray-300 focus-within:ring-2 focus-within:ring-blue-500 transition-colors flex justify-between items-start cursor-pointer",
          sizeClasses[inputSize],
          disabled && "bg-gray-100 text-gray-400 cursor-not-allowed border-gray-300",
          isOpen && "ring-2 ring-blue-500 border-transparent"
        )}
        onClick={() => !disabled && setIsOpen(!isOpen)}
        onKeyDown={handleKeyDown}
      >
        <div className="flex items-center flex-wrap gap-2 flex-1">
          {multiple && Array.isArray(value) && value.length > 0 ? (
            <>
              {Array.isArray(selectedOptions) &&
                selectedOptions
                  ?.slice(0, !isOpen && limitTags ? limitTags : selectedOptions?.length)
                  .map((option: DropdownOption) => (
                    <span
                      key={option.value}
                      className="inline-flex items-center gap-1 px-2 py-1 bg-blue-100 text-blue-700 rounded text-xs"
                    >
                      {option.label}
                      <button
                        type="button"
                        onClick={(e) => removeSelectedOption(option, e)}
                        className="hover:text-blue-900"
                      >
                        <LuX className="w-3 h-3" />
                      </button>
                    </span>
                  ))}
              {limitTags &&
                !isOpen &&
                Array.isArray(selectedOptions) &&
                selectedOptions.length > limitTags && (
                  <span className="text-sm text-gray-600">
                    +{selectedOptions.length - limitTags} more
                  </span>
                )}
            </>
          ) : !multiple && !Array.isArray(selectedOptions) && selectedOptions ? (
            <span className="text-gray-900">{selectedOptions.label}</span>
          ) : (
            <span className="text-gray-500">{placeholder}</span>
          )}
        </div>
        <div className="ml-auto flex items-center gap-1 flex-shrink-0">
          {clearable && hasAnySelection() && !disabled && (
            <button
              type="button"
              onClick={clearSelection}
              className="p-1 hover:bg-gray-100 rounded transition-colors"
            >
              <LuX className="h-4 w-4 text-gray-500 hover:text-red-600" />
            </button>
          )}
          <LuChevronDown
            className={classNames(
              "h-4 w-4 text-gray-500 transition-transform",
              isOpen && "rotate-180"
            )}
          />
        </div>
      </div>

      {isOpen && mounted && dropdownPosition && createPortal(
        <div
          ref={portalRef}
          className="fixed z-[9999] bg-white border border-gray-200 rounded-lg shadow-2xl"
          style={{
            top: `${dropdownPosition.top}px`,
            left: `${dropdownPosition.left}px`,
            width: `${dropdownPosition.width}px`,
          }}
        >
          {/* Search Input */}
          <div className="sticky top-0 bg-white border-b border-gray-200 p-2">
            <input
              ref={inputRef}
              type="text"
              placeholder="Search..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              onKeyDown={handleKeyDown}
              className="w-full px-3 py-2 border border-gray-300 rounded-md text-sm focus:outline-none focus:ring-0 focus:border-transparent"
              autoFocus
            />
          </div>

          <div
            ref={optionsRef}
            className="overflow-auto"
            style={{ maxHeight: `calc(${maxHeight} - 44px)` }}
          >
            {loading ? (
              <div className="px-4 py-3 text-sm text-gray-500 text-center">
                Loading...
              </div>
            ) : filteredOptions.length === 0 ? (
              <div className="px-4 py-3 text-sm text-gray-500 text-center">
                {noResultsText}
              </div>
            ) : (
              filteredOptions.map((option, index) => {
                const isSelected = isOptionSelected(option);
                return (
                  <div
                    key={option.value}
                    className={classNames(
                      "px-4 py-3 cursor-pointer transition-colors flex items-center justify-between",
                      {
                        "bg-blue-600 text-white": index === highlightedIndex,
                        "hover:bg-gray-100": index !== highlightedIndex && !option.disabled,
                        "opacity-50 cursor-not-allowed": option.disabled,
                        "bg-blue-50": isSelected && index !== highlightedIndex,
                      }
                    )}
                    onClick={() => handleOptionClick(option)}
                  >
                    <div className="flex-1">
                      <div
                        className={classNames("text-sm font-medium", {
                          "text-white": index === highlightedIndex,
                          "text-gray-900": index !== highlightedIndex,
                        })}
                      >
                        {option.label}
                      </div>
                      {option.subtitle && (
                        <div
                          className={classNames("text-xs", {
                            "text-blue-100": index === highlightedIndex,
                            "text-gray-600": index !== highlightedIndex,
                          })}
                        >
                          {option.subtitle}
                        </div>
                      )}
                    </div>
                    {isSelected && (
                      <LuCheck
                        className={classNames("h-4 w-4 flex-shrink-0", {
                          "text-white": index === highlightedIndex,
                          "text-blue-600": index !== highlightedIndex,
                        })}
                      />
                    )}
                  </div>
                );
              })
            )}
          </div>
        </div>,
        document.body
      )}

      {error && <div className="mt-1 text-sm text-red-600">{error}</div>}
    </div>
  );
}
