"use client";

import { SelectHTMLAttributes } from "react";
import classNames from "classnames";

interface SelectFieldProps extends SelectHTMLAttributes<HTMLSelectElement> {
  label?: string;
  error?: string;
  helperText?: string;
  options?: { value: string; label: string }[];
  containerClassName?: string;
  labelClassName?: string;
  selectClassName?: string;
  errorClassName?: string;
}

export default function SelectField({
  label,
  error,
  helperText,
  options = [],
  containerClassName,
  labelClassName,
  selectClassName,
  errorClassName,
  children,
  id,
  ...props
}: SelectFieldProps) {
  // Default styles
  const defaultContainerClassName = "w-full flex flex-col space-y-1";
  const defaultLabelClassName = "text-sm font-medium text-gray-700 cursor-pointer";
  const defaultSelectClassName =
    "w-full rounded-lg bg-white border border-gray-300 outline-1 outline-gray-300 ring-gray-300 disabled:border-gray-200 focus:outline-2 focus:ring-2 focus:ring-blue-500 focus:outline-blue-500 transition px-3 py-2";
  const defaultErrorClassName = "text-sm text-red-600";
  const defaultHelperClassName = "text-xs text-gray-500";

  const finalSelectClassName = classNames(
    defaultSelectClassName,
    {
      "border-red-500 focus:ring-red-500 focus:outline-red-500": error,
    },
    selectClassName
  );

  return (
    <div className={containerClassName || defaultContainerClassName}>
      {label && (
        <label htmlFor={id} className={labelClassName || defaultLabelClassName}>
          {label}
        </label>
      )}
      <select id={id} className={finalSelectClassName} {...props}>
        {children}
        {options.length > 0 &&
          options.map((option) => (
            <option key={option.value} value={option.value} className="hover:bg-primary-500">
              {option.label}
            </option>
          ))}
      </select>
      {error && (
        <p className={errorClassName || defaultErrorClassName}>{error}</p>
      )}
      {helperText && !error && (
        <p className={defaultHelperClassName}>{helperText}</p>
      )}
    </div>
  );
}
