"use client";

/* eslint-disable @typescript-eslint/no-explicit-any */
import { Fragment, useState } from "react";
import classNames from "classnames";

interface Column<T> {
  key: keyof T;
  label: string;
  render?: (value: any, row: T) => React.ReactNode;
  invisible?: boolean;
  sortable?: boolean;
}

type TableVariant = "dark" | "primary" | "secondary" | "accent" | "default";

interface TableProps<T> {
  data: T[];
  columns: Column<T>[];
  onRowClick?: (row: T) => void;
  className?: string;
  variant?: TableVariant;
  rowClassName?: string;
}

// Define styles based on variants - using theme colors from globals.css
const variantStyles: Record<TableVariant, { 
  header: string; 
  rowEven: string; 
  rowOdd: string; 
  hover: string; 
}> = {
  dark: {
    header: "bg-gray-900 text-white",
    rowEven: "bg-white",
    rowOdd: "bg-gray-50",
    hover: "hover:bg-gray-100",
  },
  primary: {
    header: "bg-primary-600 text-white",
    rowEven: "bg-primary-50",
    rowOdd: "bg-white",
    hover: "hover:bg-primary-100",
  },
  secondary: {
    header: "bg-secondary-600 text-white",
    rowEven: "bg-secondary-50",
    rowOdd: "bg-white",
    hover: "hover:bg-secondary-100",
  },
  accent: {
    header: "bg-primary-700 text-white",
    rowEven: "bg-primary-100",
    rowOdd: "bg-white",
    hover: "hover:bg-primary-200",
  },
  default: {
    header: "bg-primary-100 border-b-2 border-primary-300",
    rowEven: "bg-white",
    rowOdd: "bg-gray-50",
    hover: "hover:bg-primary-50",
  },
};

export default function Table<T>({
  data,
  columns,
  onRowClick,
  className,
  variant = "default",
  rowClassName,
}: TableProps<T>) {
  const [sortColumn, setSortColumn] = useState<keyof T | null>(null);
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("asc");

  const sortedData = data
    ? [...data].sort((a, b) => {
        if (!sortColumn) return 0;
        const aValue = a[sortColumn] as any;
        const bValue = b[sortColumn] as any;

        if (typeof aValue === "string" && typeof bValue === "string") {
          return sortOrder === "asc"
            ? aValue.localeCompare(bValue)
            : bValue.localeCompare(aValue);
        }

        return sortOrder === "asc"
          ? aValue > bValue
            ? 1
            : -1
          : aValue < bValue
          ? 1
          : -1;
      })
    : [];

  const handleSort = (column: keyof T) => {
    const columnConfig = columns.find((col) => col.key === column);
    if (!columnConfig?.sortable) return;

    if (sortColumn === column) {
      setSortOrder(sortOrder === "asc" ? "desc" : "asc");
    } else {
      setSortColumn(column);
      setSortOrder("asc");
    }
  };

  const isEven = (n: number) => n % 2 === 0;
  const styles = variantStyles[variant];

  const capitalize = (str: string) => {
    return str.charAt(0).toUpperCase() + str.slice(1);
  };

  return (
    <div className="overflow-x-auto border border-gray-200 rounded-lg">
      <table className={classNames("w-full min-w-max", className)}>
        {/* Table Header */}
        <thead className={classNames(styles.header, "sticky top-0")}>
          <tr>
            <th className="px-6 py-3 text-left text-sm font-semibold text-gray-700">No.</th>
            {columns.map(({ key, label, invisible, sortable = true }) => (
              <th
                key={String(key)}
                onClick={() => sortable && handleSort(key)}
                className={classNames(
                  "px-6 py-3 text-left text-sm font-semibold",
                  invisible && "hidden",
                  String(key) === "actions" && "text-right",
                  sortable && "cursor-pointer hover:bg-opacity-90 transition"
                )}
              >
                <div className="flex items-center gap-2">
                  {capitalize(label)}
                  {sortColumn === key && sortable ? (
                    <span className="text-xs">{sortOrder === "asc" ? "↑" : "↓"}</span>
                  ) : null}
                </div>
              </th>
            ))}
          </tr>
        </thead>

        {/* Table Body */}
        <tbody>
          {sortedData.map((row, rowIndex) => (
            <tr
              key={rowIndex}
              onClick={() => onRowClick?.(row)}
              className={classNames(
                styles.hover,
                isEven(rowIndex) ? styles.rowEven : styles.rowOdd,
                "transition text-gray-900 text-sm border-b border-gray-200",
                onRowClick && "cursor-pointer",
                rowClassName
              )}
            >
              <td className="px-6 py-4 font-medium text-gray-600">{rowIndex + 1}</td>
              {columns.map(({ key, render, invisible }) => (
                <Fragment key={String(key)}>
                  <td
                    className={classNames(
                      "px-6 py-4",
                      invisible && "hidden",
                      String(key) === "actions" && "text-right"
                    )}
                  >
                    {render ? (
                      <div
                        onClick={(e) => (String(key) === "actions" ? e.stopPropagation() : null)}
                      >
                        {render(row[key], row)}
                      </div>
                    ) : (
                      <span>{capitalize(String(row[key]) || "-")}</span>
                    )}
                  </td>
                </Fragment>
              ))}
            </tr>
          ))}
        </tbody>
      </table>

      {/* Empty State */}
      {sortedData.length === 0 && (
        <div className="text-center py-8 text-gray-500">
          <p>No data available</p>
        </div>
      )}
    </div>
  );
}
