import React from 'react';
import Card from '@/components/common/ui/card';
import Button from '@/components/common/ui/button';
import { Play, Eye, ArrowRight, Camera, Navigation } from 'lucide-react';

export function VirtualTour() {
  const tourStops = [
    {
      title: 'Science Classroom',
      description: 'Our dedicated science classroom where students conduct basic experiments and learn fundamental scientific principles through hands-on activities.',
      image: '/images/tour/science-classroom.jpg',
      duration: '2 min',
      highlights: ['Basic Lab Equipment', 'Learning Materials', 'Safe Environment']
    },
    {
      title: 'Computer Learning Room',
      description: 'A learning space with computers where students develop digital literacy skills and learn basic computer applications.',
      image: '/images/tour/computer-room.jpg',
      duration: '3 min',
      highlights: ['Computer Access', 'Digital Learning', 'Skill Development']
    },
    {
      title: 'Library & Study Area',
      description: 'Our quiet library space with books and study materials where students can read, research, and focus on their studies.',
      image: '/images/tour/library-study.jpg',
      duration: '4 min',
      highlights: ['Book Collection', 'Quiet Study', 'Learning Resources']
    },
    {
      title: 'Classrooms',
      description: 'Clean, well-organized classrooms where our dedicated teachers provide quality education to students.',
      image: '/images/tour/classrooms.jpg',
      duration: '2 min',
      highlights: ['Learning Environment', 'Teaching Materials', 'Student Desks']
    }
  ];

  return (
    <section id="virtual-tour" className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-20">
          <div className="inline-flex items-center gap-2 bg-secondary-100 text-secondary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
            📸 Our Facilities
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
            See Our
            <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent"> Learning Spaces</span>
          </h2>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto leading-relaxed mb-8">
            Take a look at our learning environments where students receive quality education. 
            Our facilities provide a safe and supportive space for academic growth.
          </p>

          {/* Main Tour Video */}
          <div className="max-w-4xl mx-auto mb-16">
            <Card className="overflow-hidden shadow-2xl border-0 bg-gradient-to-br from-primary-50 to-secondary-50 hover:shadow-3xl transition-all duration-500 group">
              <div className="aspect-video bg-gradient-to-br from-primary-600 to-secondary-600 relative overflow-hidden">
                <div className="absolute inset-0 bg-black/30"></div>
                <div className="absolute inset-0 flex items-center justify-center">
                  <div className="text-center text-white space-y-4">
                    <div className="w-20 h-20 bg-white/20 rounded-full flex items-center justify-center mx-auto backdrop-blur-sm hover:bg-white/30 transition-colors duration-300 cursor-pointer group-hover:scale-110 transform">
                      <Play className="w-8 h-8 text-white ml-1" />
                    </div>
                    <h3 className="text-2xl md:text-3xl font-bold">Facilities Overview</h3>
                    <p className="text-primary-100 max-w-md mx-auto">
                      See our classrooms, library, and learning spaces where education happens
                    </p>
                    <div className="flex items-center justify-center gap-2 text-sm text-primary-100">
                      <Eye className="w-4 h-4" />
                      <span>8 minutes • Photo tour</span>
                    </div>
                  </div>
                </div>
                
                {/* Decorative Elements */}
                <div className="absolute top-4 left-4 w-8 h-8 bg-white/10 rounded-full animate-pulse"></div>
                <div className="absolute bottom-4 right-4 w-6 h-6 bg-secondary-300/20 rounded-full animate-pulse delay-500"></div>
                <div className="absolute top-1/3 right-8 w-4 h-4 bg-primary-300/20 rounded-full animate-pulse delay-1000"></div>
              </div>
            </Card>
          </div>
        </div>

        {/* Tour Stops */}
        <div className="mb-16">
          <h3 className="text-2xl md:text-3xl font-bold text-center text-gray-900 mb-12">
            Our Learning Spaces
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {tourStops.map((stop, index) => (
              <Card key={index} className="overflow-hidden shadow-xl border-0 bg-white hover:shadow-2xl transition-all duration-500 hover:-translate-y-2 group">
                <div className="aspect-[4/3] bg-gradient-to-br from-gray-200 to-gray-300 relative overflow-hidden">
                  <div className="w-full h-full bg-gradient-to-br from-primary-500 to-secondary-500 flex items-center justify-center relative">
                    <Camera className="w-16 h-16 text-white opacity-60" />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent"></div>
                    
                    {/* Play Button */}
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="w-16 h-16 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-sm hover:bg-white/30 transition-colors duration-300 cursor-pointer group-hover:scale-110 transform">
                        <Play className="w-6 h-6 text-white ml-1" />
                      </div>
                    </div>

                    {/* Duration Badge */}
                    <div className="absolute top-4 right-4 bg-black/60 text-white px-3 py-1 rounded-full text-sm backdrop-blur-sm">
                      {stop.duration}
                    </div>

                    {/* Stop Number */}
                    <div className="absolute top-4 left-4 w-8 h-8 bg-primary-600 text-white rounded-full flex items-center justify-center font-bold text-sm">
                      {index + 1}
                    </div>
                  </div>
                </div>

                <div className="p-6 space-y-4">
                  <div>
                    <h4 className="text-xl font-bold text-gray-900 mb-2 group-hover:text-primary-600 transition-colors duration-300">
                      {stop.title}
                    </h4>
                    <p className="text-gray-600 leading-relaxed">
                      {stop.description}
                    </p>
                  </div>

                  <div>
                    <h5 className="font-semibold text-gray-900 mb-2">Highlights:</h5>
                    <div className="flex flex-wrap gap-2">
                      {stop.highlights.map((highlight, highlightIndex) => (
                        <span 
                          key={highlightIndex}
                          className="bg-primary-100 text-primary-700 px-3 py-1 rounded-full text-sm font-medium"
                        >
                          {highlight}
                        </span>
                      ))}
                    </div>
                  </div>

                  <Button 
                    variant="outlined" 
                    className="w-full border-primary-300 text-primary-700 hover:bg-primary-50 transition-colors duration-300 group"
                  >
                    View Photos
                    <ArrowRight className="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform duration-300" />
                  </Button>
                </div>
              </Card>
            ))}
          </div>
        </div>

        {/* Visit Information */}
        <div className="bg-gradient-to-br from-gray-50 to-primary-50 p-12 rounded-3xl">
          <div className="text-center mb-8">
            <h3 className="text-2xl md:text-3xl font-bold text-gray-900 mb-4">
              Visit Our School
            </h3>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              We welcome visitors to see our facilities in person and meet our dedicated educators
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            <div className="text-center">
              <div className="w-16 h-16 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Navigation className="w-8 h-8 text-white" />
              </div>
              <h4 className="text-xl font-bold text-gray-900 mb-2">Campus Visit</h4>
              <p className="text-gray-600">
                Schedule a visit to walk through our classrooms and see our learning environment firsthand
              </p>
            </div>

            <div className="text-center">
              <div className="w-16 h-16 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Eye className="w-8 h-8 text-white" />
              </div>
              <h4 className="text-xl font-bold text-gray-900 mb-2">Meet Our Teachers</h4>
              <p className="text-gray-600">
                Connect with our experienced educators and learn about our teaching approach
              </p>
            </div>

            <div className="text-center">
              <div className="w-16 h-16 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center mx-auto mb-4">
                <Camera className="w-8 h-8 text-white" />
              </div>
              <h4 className="text-xl font-bold text-gray-900 mb-2">School Information</h4>
              <p className="text-gray-600">
                Get detailed information about our programs, curriculum, and admission process
              </p>
            </div>
          </div>

          <div className="text-center mt-8">
            <Button className="bg-gradient-to-r from-primary-600 to-secondary-600 text-white px-8 py-4 font-semibold hover:shadow-lg transition-all duration-300 hover:bg-gradient-to-r hover:from-primary-700 hover:to-secondary-700">
              Schedule a Visit
            </Button>
          </div>
        </div>
      </div>
    </section>
  );
}

export default VirtualTour;
