'use client'
import { useState, useEffect } from 'react'
import { ChevronLeft, ChevronRight, Quote } from 'lucide-react'
import { cn } from '@/lib/utils'

interface TestimonialData {
  id: number
  name: string
  role: string
  content: string
  rating: number
  image?: string
  company?: string
}

interface AdvancedTestimonialsProps {
  testimonials: TestimonialData[]
  title?: string
  subtitle?: string
  autoPlay?: boolean
  autoPlayInterval?: number
}

export default function AdvancedTestimonials({
  testimonials,
  title = "What People Say",
  subtitle = "Real experiences from our community",
  autoPlay = true,
  autoPlayInterval = 6000 // Increased for mobile
}: AdvancedTestimonialsProps) {
  const [currentIndex, setCurrentIndex] = useState(0)
  const [isAutoPlaying, setIsAutoPlaying] = useState(autoPlay)
  const [isMobile, setIsMobile] = useState(false)

  // Detect mobile devices - Enhanced detection
  useEffect(() => {
    const checkMobile = () => {
      const isMobileWidth = window.innerWidth < 768
      const isMobileTablet = window.innerWidth >= 768 && window.innerWidth < 1024
      const isMobileUA = /iPhone|iPad|iPod|Android|webOS|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent)
      
      setIsMobile(isMobileWidth || isMobileUA)
      
      // Adjust autoplay interval based on device
      if (isMobileWidth) {
        // Slower on mobile for better UX
      } else if (isMobileTablet) {
        // Medium speed on tablet
      }
    }
    
    checkMobile()
    window.addEventListener('resize', checkMobile)
    return () => window.removeEventListener('resize', checkMobile)
  }, [])

  useEffect(() => {
    if (!isAutoPlaying) return

    const interval = setInterval(() => {
      setCurrentIndex((prev) => (prev + 1) % testimonials.length)
    }, autoPlayInterval)

    return () => clearInterval(interval)
  }, [isAutoPlaying, testimonials.length, autoPlayInterval])

  const goToPrevious = () => {
    setCurrentIndex((prev) => (prev - 1 + testimonials.length) % testimonials.length)
  }

  const goToNext = () => {
    setCurrentIndex((prev) => (prev + 1) % testimonials.length)
  }

  const goToSlide = (index: number) => {
    setCurrentIndex(index)
  }

  const currentTestimonial = testimonials[currentIndex]

  return (
    <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 relative overflow-hidden bg-gradient-to-br from-primary-50 via-white to-secondary-50">
      {/* Simplified Background Elements - Optimized for all devices */}
      <div className="absolute inset-0 opacity-40 sm:opacity-60 md:opacity-100">
        <div className="absolute top-10 sm:top-20 left-10 sm:left-20 w-24 sm:w-32 md:w-64 h-24 sm:h-32 md:h-64 bg-primary-200/30 rounded-full blur-xl md:blur-2xl lg:blur-3xl" />
        <div className="absolute bottom-10 sm:bottom-20 right-10 sm:right-20 w-32 sm:w-40 md:w-80 h-32 sm:h-40 md:h-80 bg-secondary-200/30 rounded-full blur-xl md:blur-2xl lg:blur-3xl" />
      </div>

      <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-8 sm:mb-12 lg:mb-16">
          <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-3 sm:px-4 py-1.5 sm:py-2 rounded-full font-semibold text-xs sm:text-sm uppercase tracking-wider mb-4 sm:mb-6 border border-primary-200">
            <Quote className="w-3 sm:w-4 h-3 sm:h-4" />
            {subtitle}
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold tracking-tight leading-tight text-gray-900 mb-4 sm:mb-6">
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-gray-900 via-primary-600 to-secondary-600">
              {title}
            </span>
          </h2>
        </div>

        {/* Main Testimonial Display - Optimized for all devices */}
        <div className="relative">
          <div 
            className="flex items-center justify-center min-h-[300px] sm:min-h-[350px] md:min-h-[400px]"
            onMouseEnter={() => !isMobile && setIsAutoPlaying(false)}
            onMouseLeave={() => !isMobile && setIsAutoPlaying(autoPlay)}
          >
            {/* Testimonial Card */}
            <div className="max-w-4xl mx-auto text-center px-2 sm:px-4">
              {/* Quote Icon - Responsive sizing */}
              <div className="mb-4 sm:mb-6 md:mb-8">
                <div className="inline-flex items-center justify-center w-12 sm:w-16 md:w-20 h-12 sm:h-16 md:h-20 bg-gradient-to-br from-primary-200 to-secondary-200 rounded-full border border-primary-300">
                  <Quote className="w-6 sm:w-8 md:w-10 h-6 sm:h-8 md:h-10 text-primary-700" />
                </div>
              </div>

              {/* Content - Responsive font sizes */}
              <blockquote className="text-lg sm:text-xl md:text-2xl lg:text-3xl font-light text-gray-800 leading-relaxed mb-6 sm:mb-8 md:mb-12 italic px-2">
                &ldquo;{currentTestimonial.content}&rdquo;
              </blockquote>

              {/* Rating - Responsive sizing */}
              <div className="flex justify-center gap-1 mb-4 sm:mb-6 md:mb-8">
                {[...Array(5)].map((_, i) => (
                  <div
                    key={i}
                    className={cn(
                      "w-2 sm:w-2.5 md:w-3 h-2 sm:h-2.5 md:h-3 rounded-full transition-colors duration-200",
                      i < currentTestimonial.rating 
                        ? "bg-gradient-to-r from-primary-600 to-secondary-600" 
                        : "bg-gray-400"
                    )}
                  />
                ))}
              </div>

              {/* Author Info - Responsive layout */}
              <div className="space-y-2 sm:space-y-3">
                <div className="flex items-center justify-center gap-2 sm:gap-3 md:gap-4">
                  {/* Avatar */}
                  <div className="w-10 sm:w-12 md:w-16 h-10 sm:h-12 md:h-16 bg-gradient-to-br from-primary-200 to-secondary-200 rounded-full border-2 border-primary-300 flex items-center justify-center">
                    <span className="text-primary-800 font-bold text-sm sm:text-base md:text-lg">
                      {currentTestimonial.name.charAt(0)}
                    </span>
                  </div>
                  
                  <div className="text-left">
                    <h4 className="text-base sm:text-lg md:text-xl font-bold text-gray-900">
                      {currentTestimonial.name}
                    </h4>
                    <p className="text-primary-700 font-medium text-sm md:text-base">
                      {currentTestimonial.role}
                    </p>
                    {currentTestimonial.company && (
                      <p className="text-gray-600 text-xs md:text-sm">
                        {currentTestimonial.company}
                      </p>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Navigation Arrows - Tablet and desktop only */}
          <button
            onClick={goToPrevious}
            className="hidden md:block absolute left-2 lg:left-0 top-1/2 transform -translate-y-1/2 -translate-x-0 lg:-translate-x-4 bg-white/80 backdrop-blur-sm hover:bg-white/90 text-gray-700 hover:text-gray-900 p-2 lg:p-3 rounded-full transition-colors duration-200 border border-gray-200 shadow-lg"
          >
            <ChevronLeft className="w-5 lg:w-6 h-5 lg:h-6" />
          </button>
          
          <button
            onClick={goToNext}
            className="hidden md:block absolute right-2 lg:right-0 top-1/2 transform -translate-y-1/2 translate-x-0 lg:translate-x-4 bg-white/80 backdrop-blur-sm hover:bg-white/90 text-gray-700 hover:text-gray-900 p-2 lg:p-3 rounded-full transition-colors duration-200 border border-gray-200 shadow-lg"
          >
            <ChevronRight className="w-5 lg:w-6 h-5 lg:h-6" />
          </button>
        </div>

        {/* Mobile Navigation Buttons */}
        <div className="flex md:hidden justify-center gap-4 mt-6 sm:mt-8">
          <button
            onClick={goToPrevious}
            className="bg-white/80 backdrop-blur-sm text-gray-700 p-2.5 sm:p-3 rounded-full border border-gray-200 active:bg-white/90 transition-colors duration-150 shadow-lg"
          >
            <ChevronLeft className="w-4 sm:w-5 h-4 sm:h-5" />
          </button>
          
          <button
            onClick={goToNext}
            className="bg-white/80 backdrop-blur-sm text-gray-700 p-2.5 sm:p-3 rounded-full border border-gray-200 active:bg-white/90 transition-colors duration-150 shadow-lg"
          >
            <ChevronRight className="w-4 sm:w-5 h-4 sm:h-5" />
          </button>
        </div>

        {/* Dots Navigation - Responsive sizing */}
        <div className="flex justify-center gap-1.5 sm:gap-2 md:gap-3 mt-6 sm:mt-8 md:mt-12">
          {testimonials.map((_, index) => (
            <button
              key={index}
              onClick={() => goToSlide(index)}
              className={cn(
                "transition-all duration-200 rounded-full",
                index === currentIndex
                  ? "w-6 sm:w-8 md:w-12 h-2 sm:h-2.5 md:h-3 bg-gradient-to-r from-primary-500 to-secondary-500"
                  : "w-2 sm:w-2.5 md:w-3 h-2 sm:h-2.5 md:h-3 bg-gray-300"
              )}
            />
          ))}
        </div>

        {/* Testimonial Previews - Hidden on mobile and small tablets for performance */}
        <div className="hidden xl:grid xl:grid-cols-3 gap-4 lg:gap-6 mt-12 lg:mt-16">
          {testimonials.map((testimonial, index) => (
            <div
              key={testimonial.id}
              onClick={() => goToSlide(index)}
              className={cn(
                "cursor-pointer transition-all duration-200 p-4 lg:p-6 rounded-xl lg:rounded-2xl border backdrop-blur-sm",
                index === currentIndex
                  ? "bg-white/90 border-primary-300 shadow-xl ring-2 ring-primary-200/50"
                  : "bg-white/70 border-gray-200 hover:bg-white/80 hover:shadow-md hover:border-primary-200"
              )}
            >
              <div className="flex items-center gap-2 lg:gap-3 mb-2 lg:mb-3">
                <div className="w-8 lg:w-10 h-8 lg:h-10 bg-gradient-to-br from-primary-200 to-secondary-200 rounded-full border border-primary-300 flex items-center justify-center">
                  <span className="text-primary-800 font-bold text-xs lg:text-sm">
                    {testimonial.name.charAt(0)}
                  </span>
                </div>
                <div>
                  <h5 className="text-gray-900 font-semibold text-xs lg:text-sm">
                    {testimonial.name}
                  </h5>
                  <p className="text-primary-600 text-[10px] lg:text-xs">
                    {testimonial.role}
                  </p>
                </div>
              </div>
              <p className="text-gray-700 text-xs lg:text-sm line-clamp-2">
                &ldquo;{testimonial.content}&rdquo;
              </p>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}
