'use client'
import { useEffect, useState, useRef } from 'react'
import { cn } from '@/lib/utils'

interface StatItemProps {
  value: number
  label: string
  suffix?: string
  prefix?: string
  description?: string
  color?: 'primary' | 'secondary' | 'accent'
  delay?: number
}

interface NextGenStatsProps {
  stats: StatItemProps[]
  title?: string
  subtitle?: string
  className?: string
}

function StatCard({ 
  value, 
  label, 
  suffix = '', 
  prefix = '', 
  description,
  color = "primary",
  delay = 0
}: StatItemProps) {
  const [count, setCount] = useState(0)
  const [isVisible, setIsVisible] = useState(false)
  const ref = useRef<HTMLDivElement>(null)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
          const timer = setTimeout(() => {
            let start = 0
            const end = value
            const duration = 2000
            const increment = end / (duration / 16)

            const counter = setInterval(() => {
              start += increment
              if (start >= end) {
                setCount(end)
                clearInterval(counter)
              } else {
                setCount(Math.floor(start))
              }
            }, 16)

            return () => clearInterval(counter)
          }, delay)

          return () => clearTimeout(timer)
        }
      },
      { threshold: 0.3 }
    )

    if (ref.current) {
      observer.observe(ref.current)
    }

    return () => observer.disconnect()
  }, [value, delay])

  const colorClasses = {
    primary: {
      bg: 'from-primary-500/10 to-primary-600/20',
      border: 'border-primary-300/30',
      accent: 'text-primary-600',
      glow: 'shadow-primary-500/20'
    },
    secondary: {
      bg: 'from-secondary-500/10 to-secondary-600/20',
      border: 'border-secondary-300/30',
      accent: 'text-secondary-600',
      glow: 'shadow-secondary-500/20'
    },
    accent: {
      bg: 'from-emerald-500/10 to-emerald-600/20',
      border: 'border-emerald-300/30',
      accent: 'text-emerald-600',
      glow: 'shadow-emerald-500/20'
    }
  }

  return (
    <div 
      ref={ref}
      className={cn(
        "group relative overflow-hidden",
        "bg-gradient-to-br", colorClasses[color].bg,
        "backdrop-blur-sm border", colorClasses[color].border,
        "rounded-2xl p-8 transition-all duration-500",
        "hover:scale-105 hover:shadow-2xl", colorClasses[color].glow,
        isVisible ? 'translate-y-0 opacity-100' : 'translate-y-8 opacity-0'
      )}
      style={{ transitionDelay: `${delay}ms` }}
    >
      {/* Background decoration */}
      <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-white/10 to-transparent rounded-full blur-3xl transform translate-x-8 -translate-y-8 group-hover:scale-150 transition-transform duration-700" />
      
      {/* Content */}
      <div className="relative z-10 flex justify-center items-center flex-col">
        {/* Number */}
        <div className={cn(
          "text-4xl md:text-5xl font-bold mb-4 leading-none",
          colorClasses[color].accent
        )}>
          {prefix}{count.toLocaleString()}{suffix}
        </div>
        
        {/* Label */}
        <h3 className="text-xl font-bold text-gray-900 mb-2">
          {label}
        </h3>
        
        {/* Description */}
        {description && (
          <p className="text-gray-600 text-sm leading-relaxed">
            {description}
          </p>
        )}
      </div>

      {/* Hover effect line */}
      <div className={cn(
        "absolute bottom-0 left-0 h-1 w-0 group-hover:w-full transition-all duration-500",
        `bg-gradient-to-r ${colorClasses[color].bg}`
      )} />
    </div>
  )
}

export default function NextGenStats({ 
  stats, 
  title, 
  subtitle,
  className 
}: NextGenStatsProps) {
  return (
    <section className={cn("py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 relative overflow-hidden", className)}>
      {/* Background Elements */}
      <div className="absolute inset-0 bg-gradient-to-br from-gray-50 via-white to-primary-50/30" />
      <div className="absolute top-20 left-10 w-72 h-72 bg-primary-100/30 rounded-full blur-3xl" />
      <div className="absolute bottom-20 right-10 w-96 h-96 bg-secondary-100/20 rounded-full blur-3xl" />
      
      <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        {(title || subtitle) && (
          <div className="text-center mb-20">
            {subtitle && (
              <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full font-semibold text-sm uppercase tracking-wider mb-6">
                <div className="w-2 h-2 bg-primary-500 rounded-full animate-pulse" />
                {subtitle}
              </div>
            )}
            {title && (
              <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold text-gray-900 mb-6">
                <span className="text-transparent bg-clip-text bg-gradient-to-r from-gray-900 via-primary-600 to-secondary-600">
                  {title}
                </span>
              </h2>
            )}
          </div>
        )}

        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {stats.map((stat, index) => (
            <StatCard 
              key={index}
              {...stat}
              delay={index * 150}
            />
          ))}
        </div>
      </div>
    </section>
  )
}
