"use client";
import { useEffect, useState } from "react";
import {
  ArrowRight,
  ChevronLeft,
  ChevronRight,
  Play,
  Sparkles,
} from "lucide-react";
import Image from "next/image";
import Button from "@/components/common/ui/button";
import { usePublicHeroSlides } from "@/hooks/useHeroSlides";

// Swiper imports
import { Swiper, SwiperSlide } from "swiper/react";
import { Navigation, Pagination, Autoplay, Thumbs } from "swiper/modules";
import type { Swiper as SwiperType } from "swiper";
import "swiper/css";
import "swiper/css/navigation";
import "swiper/css/pagination";
import "swiper/css/thumbs";

interface UltraModernHeroProps {
  title: string;
  subtitle: string;
  description: string;
  primaryCTA: { text: string; href: string };
  secondaryCTA: { text: string; href: string };
}

// Fallback images if no slides are loaded from the database
const defaultGalleryImages = [
  {
    src: "/images/home/hom1-edited.avif",
    alt: "Setaragan Zirak School Main Building",
    title: "Our Campus",
    description: "Modern learning environment",
  },
  {
    src: "/images/home/hom2-768x1024.avif",
    alt: "Students Learning Together",
    title: "Quality Education",
    description: "Empowering young minds",
  },
  {
    src: "/images/home/raihana-768x1024.avif",
    alt: "Happy Student",
    title: "Our Students",
    description: "Building futures through education",
  },
];

export default function UltraModernHero({
  subtitle,
  description,
  primaryCTA,
  secondaryCTA,
}: UltraModernHeroProps) {
  const [isVisible, setIsVisible] = useState(false);
  const [isMobile, setIsMobile] = useState(false);
  const [thumbsSwiper, setThumbsSwiper] = useState<SwiperType | null>(null);
  
  // Fetch hero slides from API
  const { data: heroSlides, isLoading } = usePublicHeroSlides();

  // Use API slides if available, otherwise use default images
  const galleryImages = heroSlides && heroSlides.length > 0
    ? heroSlides.map(slide => ({
        src: slide.imageUrl,
        alt: slide.alt,
        title: slide.title,
        description: slide.description || '',
      }))
    : defaultGalleryImages;

  useEffect(() => {
    const checkMobile = () => {
      setIsMobile(window.innerWidth < 768);
    };
    checkMobile();
    window.addEventListener("resize", checkMobile);
    const timer = setTimeout(() => setIsVisible(true), 200);
    return () => {
      clearTimeout(timer);
      window.removeEventListener("resize", checkMobile);
    };
  }, []);

  return (
    <div className="relative min-h-screen flex items-center justify-center overflow-hidden bg-gradient-to-br from-primary-400 via-primary-500 to-secondary-600 py-8 sm:py-12 lg:py-16">
      {/* Background Decorations */}
      <div className="absolute inset-0">
        <div className="absolute inset-0 opacity-20 md:opacity-30 bg-[radial-gradient(circle_at_center,rgba(255,255,255,0.2),transparent_40%)]" />
        {!isMobile && (
          <>
            <div className="absolute top-20 left-10 w-32 h-32 bg-white/10 rounded-full blur-xl animate-pulse" />
            <div className="absolute top-40 right-20 w-48 h-48 bg-white/5 rounded-full blur-2xl animate-bounce duration-3000" />
            <div className="absolute bottom-20 left-1/4 w-24 h-24 bg-secondary-900/20 rounded-full blur-lg animate-pulse delay-1000" />
          </>
        )}
      </div>

      {/* Content */}
      <div className="relative z-10 max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 w-full">
        <div className="grid grid-cols-1 lg:grid-cols-12 gap-6 sm:gap-8 lg:gap-12 xl:gap-16 items-center">
          {/* Left Section */}
          <div className="lg:col-span-5 text-left">
            <div
              className={`flex items-center gap-2 sm:gap-3 mb-4 sm:mb-6 transition-opacity duration-500 ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              <div className="flex items-center gap-2 bg-white/10 backdrop-blur-sm px-3 sm:px-4 py-1.5 sm:py-2 rounded-full border border-white/20">
                <Sparkles className="w-3 sm:w-4 h-3 sm:h-4 text-white" />
                <span className="text-white font-semibold text-xs sm:text-sm tracking-wide uppercase">
                  Excellence Since 2019
                </span>
              </div>
            </div>

            <h1
              className={`text-4xl sm:text-5xl md:text-6xl font-extrabold tracking-tight leading-none mb-4 sm:mb-6 lg:mb-8 bg-clip-text text-transparent bg-gradient-to-r from-white to-primary-300 transition-opacity duration-700 strapline-words ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              Today&apos;s Stars, <br />
              <span className="text-transparent ">
                Tomorrow&apos;s Leaders
              </span>
            </h1>

            <p
              className={`text-base sm:text-lg md:text-xl lg:text-2xl text-white/90 leading-relaxed mb-6 sm:mb-8 lg:mb-10 max-w-xl transition-opacity duration-900 ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              {description}
            </p>

            {/* Stats */}
            <div
              className={`grid grid-cols-3 gap-4 sm:gap-6 mb-6 sm:mb-8 lg:mb-10 max-w-sm transition-opacity duration-1000 ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              <div className="text-center">
                <p className="text-white font-black text-2xl sm:text-3xl md:text-4xl">
                  6+
                </p>
                <p className="text-white/70 text-xs sm:text-sm font-medium">
                  Years
                </p>
              </div>
              <div className="text-center">
                <p className="text-white font-black text-2xl sm:text-3xl md:text-4xl">
                  500+
                </p>
                <p className="text-white/70 text-xs sm:text-sm font-medium">
                  Students
                </p>
              </div>
              <div className="text-center">
                <p className="text-white font-black text-2xl sm:text-3xl md:text-4xl">
                  95%
                </p>
                <p className="text-white/70 text-xs sm:text-sm font-medium">
                  Success
                </p>
              </div>
            </div>

            {/* CTA */}
            <div
              className={`flex flex-col sm:flex-row gap-2 sm:gap-3 lg:gap-4 transition-opacity duration-1000 ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              <Button
                variant="contained"
                size="lg"
                className="group bg-transparent! bg-gradient-to-r from-primary-600 to-secondary-500 hover:from-primary-700 hover:to-secondary-600 text-white border-0 shadow-2xl w-full sm:w-auto"
              >
                <span className="flex items-center justify-center gap-1.5 sm:gap-2">
                  {primaryCTA.text}
                  <ArrowRight className="w-3 sm:w-4 h-3 sm:h-4" />
                </span>
              </Button>
              <Button
                variant="outlined"
                size="lg"
                className="group border-2 border-white! text-white! hover:bg-white/10! w-full sm:w-auto"
              >
                <span className="flex items-center justify-center gap-1.5 sm:gap-2">
                  <Play className="w-3 sm:w-4 h-3 sm:h-4" />
                  {secondaryCTA.text}
                </span>
              </Button>
            </div>
          </div>

          {/* Right Section (Gallery with Swiper + Thumbnails) */}
          <div className="lg:col-span-7 mt-6 sm:mt-8 lg:mt-0">
            <div
              className={`relative transition-opacity duration-1000 ${
                isVisible ? "opacity-100" : "opacity-0"
              }`}
            >
              <div className="relative w-full max-w-2xl mx-auto bg-gradient-to-br from-white/10 to-white/5 backdrop-blur-xl rounded-2xl sm:rounded-3xl p-3 sm:p-4 lg:p-5 shadow-2xl">
                {/* Gallery Header */}
                <div className="flex items-center justify-between mb-3 sm:mb-4 lg:mb-6">
                  <div className="flex items-center gap-2">
                    <div className="w-2 sm:w-3 h-2 sm:h-3 bg-red-500 rounded-full"></div>
                    <div className="w-2 sm:w-3 h-2 sm:h-3 bg-yellow-500 rounded-full"></div>
                    <div className="w-2 sm:w-3 h-2 sm:h-3 bg-green-500 rounded-full"></div>
                  </div>
                  <div className="text-white/70 text-xs sm:text-sm font-medium">
                    School Gallery
                  </div>
                </div>

                {/* Main Swiper */}
                <Swiper
                  modules={[Navigation, Pagination, Autoplay, Thumbs]}
                  navigation={{
                    nextEl: ".custom-next",
                    prevEl: ".custom-prev",
                  }}
                  pagination={{ clickable: true }}
                  autoplay={{ delay: 5000, disableOnInteraction: false }}
                  loop
                  thumbs={{ swiper: thumbsSwiper }}
                  className="rounded-xl sm:rounded-2xl overflow-hidden mb-3 sm:mb-4 lg:mb-6"
                >
                  {galleryImages.map((image, index) => (
                    <SwiperSlide key={index}>
                      <div className="relative aspect-[4/3]">
                        <Image
                          src={image.src}
                          alt={image.alt}
                          fill
                          className="object-cover"
                          sizes="(max-width: 640px) 100vw, (max-width: 1024px) 50vw, 40vw"
                          priority={index === 0}
                        />
                        <div className="absolute inset-0 bg-gradient-to-t from-black/50 via-transparent to-transparent" />
                        <div className="absolute bottom-3 left-3 right-3 text-white">
                          <h3 className="text-base sm:text-lg md:text-xl font-bold">
                            {image.title}
                          </h3>
                          <p className="text-xs sm:text-sm opacity-90">
                            {image.description}
                          </p>
                        </div>
                      </div>
                    </SwiperSlide>
                  ))}
                  <button className="custom-prev absolute top-1/2 left-6 z-10 -translate-y-1/2 bg-black/40 hover:bg-black/60 text-white p-2 sm:p-3 rounded-full transition">
                    <ChevronLeft className="w-5 h-5" />
                  </button>
                  <button className="custom-next absolute top-1/2   right-6 z-10 -translate-y-1/2 bg-black/40 hover:bg-black/60 text-white p-2 sm:p-3 rounded-full transition">
                    <ChevronRight className="w-5 h-5" />
                  </button>
                </Swiper>

                {/* Thumbnail Swiper */}
                <Swiper
                  modules={[Thumbs]}
                  onSwiper={setThumbsSwiper}
                  slidesPerView={3}
                  spaceBetween={12}
                  watchSlidesProgress
                  className="grid grid-cols-3 gap-2 sm:gap-3"
                >
                  {galleryImages.map((image, index) => (
                    <SwiperSlide key={index}>
                      <div className="relative overflow-hidden rounded-lg sm:rounded-xl aspect-[4/3] cursor-pointer">
                        <Image
                          src={image.src}
                          alt={image.alt}
                          fill
                          className="object-cover"
                          sizes="(max-width: 640px) 25vw, (max-width: 1024px) 15vw, 12vw"
                        />
                        <div className="absolute inset-0 bg-gradient-to-t from-black/30 to-transparent"></div>
                        <div className="absolute bottom-1 left-1 right-1 text-white">
                          <p className="text-xs font-semibold truncate">
                            {image.title}
                          </p>
                        </div>
                      </div>
                    </SwiperSlide>
                  ))}
                </Swiper>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
