"use client";

import React from "react";
import Link from "next/link";
import Badge from "@/components/common/ui/badge";
import Card from "@/components/common/ui/card";
import {
  Calendar,
  Clock,
  User,
  ArrowLeft,
  ArrowRight,
  Share2,
  Bookmark,
  Eye,
  MessageCircle,
  Mail,
  GraduationCap,
  Phone,
} from "lucide-react";
import { NewsArticle, getRelatedArticles } from "@/lib/mockNewsData";
import { SafeImage } from "@/components/common/ui";

interface NewsArticlePageProps {
  article: NewsArticle;
}

export default function NewsArticlePage({ article }: NewsArticlePageProps) {
  const relatedArticles = getRelatedArticles(article, 3);

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "long",
      day: "numeric",
    });
  };

  return (
    <main className="min-h-screen">
      {/* Hero Section */}
      <section className="relative bg-gradient-to-br from-primary-50 via-white to-secondary-50 py-12 sm:py-16 lg:py-20 overflow-hidden">
        {/* Floating Background Elements */}
        <div className="absolute top-20 left-10 w-20 h-20 bg-primary-400/20 rounded-full blur-xl animate-pulse"></div>
        <div className="absolute bottom-20 right-10 w-32 h-32 bg-secondary-400/20 rounded-full blur-xl animate-pulse delay-1000"></div>

        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Back Navigation */}
          <div className="mb-8 lg:mb-12">
            <Link
              href="/news"
              className="inline-flex items-center gap-2 text-primary-600 hover:text-primary-700 font-medium transition-colors duration-200 group"
            >
              <ArrowLeft className="w-4 h-4 group-hover:-translate-x-1 transition-transform duration-200" />
              Back to News
            </Link>
          </div>

          <div className="grid lg:grid-cols-3 gap-8 lg:gap-12">
            {/* Article Content */}
            <div className="lg:col-span-2">
              {/* Article Meta */}
              <div className="flex flex-wrap items-center gap-4 text-sm text-gray-600 mb-6">
                <Badge variant="secondary">{article.category}</Badge>
                <div className="flex items-center gap-1">
                  <Calendar className="w-4 h-4" />
                  {formatDate(article.date)}
                </div>
                <div className="flex items-center gap-1">
                  <Clock className="w-4 h-4" />
                  {article.readTime} min read
                </div>
                <div className="flex items-center gap-1">
                  <User className="w-4 h-4" />
                  {article.author}
                </div>
              </div>

              {/* Article Title */}
              <h1 className="text-3xl sm:text-4xl lg:text-5xl font-bold leading-tight text-gray-900 mb-6">
                {article.title}
              </h1>

              {/* Article Excerpt */}
              <p className="text-xl text-gray-600 leading-relaxed mb-8">
                {article.excerpt}
              </p>

              {/* Tags */}
              <div className="flex flex-wrap gap-2 mb-8">
                {article.tags.map((tag) => (
                  <Badge key={tag} variant="outline" className="text-xs">
                    #{tag}
                  </Badge>
                ))}
              </div>

              {/* Social Actions */}
              <div className="flex items-center gap-4 mb-8 pb-8 border-b border-gray-200">
                <button className="inline-flex items-center gap-2 text-gray-600 hover:text-primary-600 transition-colors duration-200">
                  <Share2 className="w-5 h-5" />
                  Share
                </button>
                <button className="inline-flex items-center gap-2 text-gray-600 hover:text-primary-600 transition-colors duration-200">
                  <Bookmark className="w-5 h-5" />
                  Save
                </button>
                <div className="flex items-center gap-2 text-gray-600">
                  <Eye className="w-5 h-5" />
                  247 views
                </div>
              </div>
            </div>

            {/* Sidebar */}
            <div className="lg:col-span-1">
              {/* Author Info */}
              <Card className="p-6 mb-8">
                <div className="flex items-center gap-4 mb-4">
                  <div className="w-12 h-12 bg-primary-100 rounded-full flex items-center justify-center">
                    <User className="w-6 h-6 text-primary-600" />
                  </div>
                  <div>
                    <h4 className="font-semibold text-gray-900">
                      {article.author}
                    </h4>
                    <p className="text-sm text-gray-600">Staff Writer</p>
                  </div>
                </div>
                <p className="text-sm text-gray-600 mb-4">
                  Experienced educator and writer covering school news, academic
                  achievements, and community events.
                </p>
                <div className="flex gap-2">
                  <button className="flex-1 bg-primary-600 hover:bg-primary-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors duration-200">
                    More from {article.author}
                  </button>
                  {/* <button className="p-2 border border-gray-200 hover:border-gray-300 rounded-lg transition-colors duration-200">
                    <MessageCircle className="w-4 h-4 text-gray-600" />
                  </button> */}
                </div>
              </Card>

              {/* Quick Actions */}
              <Card className="p-6">
                <h4 className="font-semibold text-gray-900 mb-4">
                  Quick Actions
                </h4>
                <div className="space-y-3">
                  <Link
                    href="/contact"
                    className="flex items-center gap-3 text-gray-700 hover:text-primary-600 transition-colors duration-200"
                  >
                    <Mail className="w-5 h-5" />
                    Contact School
                  </Link>
                  <Link
                    href="/admissions"
                    className="flex items-center gap-3 text-gray-700 hover:text-primary-600 transition-colors duration-200"
                  >
                    <GraduationCap className="w-5 h-5" />
                    Learn About Admissions
                  </Link>
                  <Link
                    href="/facilities"
                    className="flex items-center gap-3 text-gray-700 hover:text-primary-600 transition-colors duration-200"
                  >
                    <Eye className="w-5 h-5" />
                    Tour Our Facilities
                  </Link>
                </div>
              </Card>
            </div>
          </div>
        </div>
      </section>

      {/* Featured Image */}
      <section className="py-8 bg-white">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="relative aspect-[21/9] rounded-2xl overflow-hidden shadow-2xl">
            <SafeImage
              src={article.featuredImage}
              alt={article.title}
              fill
              priority
              className="object-cover"
            />
          </div>
        </div>
      </section>

      {/* Article Content */}
      <section className="py-12 sm:py-16 lg:py-20 bg-white">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="max-w-4xl mx-auto">
            <div
              className="prose prose-lg max-w-none text-gray-700 leading-relaxed"
              dangerouslySetInnerHTML={{ __html: article.content }}
            />
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-16 bg-gradient-to-r from-primary-600 to-secondary-600">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center text-white">
            <h3 className="text-2xl sm:text-3xl font-bold mb-4">
              Interested in Joining Our School Community?
            </h3>
            <p className="text-xl text-primary-100 mb-8 max-w-3xl mx-auto">
              Discover more about our programs, facilities, and educational
              opportunities. Get in touch with us to learn how we can support
              your child&apos;s academic journey.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link
                href="/admissions"
                className="inline-flex items-center justify-center gap-2 bg-white text-primary-600 px-8 py-4 rounded-xl font-bold text-lg hover:bg-primary-50 transition-colors duration-200"
              >
                <GraduationCap className="w-6 h-6" />
                Apply Now
              </Link>
              <Link
                href="/contact"
                className="inline-flex items-center justify-center gap-2 border-2 border-white text-white px-8 py-4 rounded-xl font-bold text-lg hover:bg-white hover:text-primary-600 transition-colors duration-200"
              >
                <Phone className="w-6 h-6" />
                Contact Us
              </Link>
            </div>
          </div>
        </div>
      </section>

      {/* Related Articles */}
      {relatedArticles.length > 0 && (
        <section className="py-16 sm:py-20 bg-gray-50">
          <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-2xl sm:text-3xl font-bold text-gray-900 mb-4">
                Related Articles
              </h2>
              <p className="text-lg text-gray-600">
                Continue reading more news and updates from our school
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {relatedArticles.map((relatedArticle) => (
                <RelatedArticleCard
                  key={relatedArticle.id}
                  article={relatedArticle}
                />
              ))}
            </div>

            <div className="text-center mt-12">
              <Link
                href="/news"
                className="inline-flex items-center gap-2 bg-primary-600 hover:bg-primary-700 text-white px-8 py-4 rounded-xl font-semibold transition-all duration-200 hover:gap-3"
              >
                View All News
                <ArrowRight className="w-5 h-5" />
              </Link>
            </div>
          </div>
        </section>
      )}
    </main>
  );
}

// Related Article Card Component
function RelatedArticleCard({ article }: { article: NewsArticle }) {
  return (
    <Card className="overflow-hidden hover:shadow-xl transition-all duration-300 group hover:scale-105">
      <div className="relative aspect-[4/3] overflow-hidden">
        <SafeImage
          src={article.featuredImage}
          alt={article.title}
          fill
          className="object-cover group-hover:scale-110 transition-transform duration-500"
        />
        <div className="absolute top-4 left-4">
          <Badge variant="secondary">{article.category}</Badge>
        </div>
      </div>

      <div className="p-6">
        <div className="flex items-center gap-2 text-sm text-gray-500 mb-3">
          <Calendar className="w-4 h-4" />
          {new Date(article.date).toLocaleDateString("en-US", {
            month: "short",
            day: "numeric",
          })}
          <span>•</span>
          <Clock className="w-4 h-4" />
          {article.readTime} min
        </div>

        <h3 className="text-lg font-bold text-gray-900 mb-3 line-clamp-2 group-hover:text-primary-600 transition-colors duration-300">
          {article.title}
        </h3>

        <p className="text-gray-600 text-sm leading-relaxed mb-4 line-clamp-2">
          {article.excerpt}
        </p>

        <Link
          href={`/news/${article.slug}`}
          className="inline-flex items-center gap-2 text-primary-600 hover:text-primary-700 font-semibold text-sm group-hover:gap-3 transition-all duration-300"
        >
          Read Article
          <ArrowRight className="w-4 h-4" />
        </Link>
      </div>
    </Card>
  );
}
