'use client';

import React, { useState } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import Container from '@/components/common/ui/container';
import Badge from '@/components/common/ui/badge';
import Card from '@/components/common/ui/card';
import { 
  Calendar, 
  Clock, 
  User, 
  Search, 
  Filter,
  ArrowRight,
  BookOpen,
  Newspaper
} from 'lucide-react';
import { mockNewsData, newsCategories, getArticlesByCategory, NewsArticle } from '@/lib/mockNewsData';
import NewsletterSignup from './NewsletterSignup';
import { cn } from '@/lib/utils';

export default function NewsPage() {
  const [selectedCategory, setSelectedCategory] = useState<string>('All');
  const [searchQuery, setSearchQuery] = useState<string>('');
  const [displayedArticles, setDisplayedArticles] = useState<NewsArticle[]>(mockNewsData);

  const handleCategoryChange = (category: string) => {
    setSelectedCategory(category);
    const filtered = getArticlesByCategory(category);
    setDisplayedArticles(filtered);
  };

  const handleSearch = (query: string) => {
    setSearchQuery(query);
    let filtered = selectedCategory === 'All' ? mockNewsData : getArticlesByCategory(selectedCategory);
    
    if (query.trim()) {
      filtered = filtered.filter(article => 
        article.title.toLowerCase().includes(query.toLowerCase()) ||
        article.excerpt.toLowerCase().includes(query.toLowerCase()) ||
        article.tags.some(tag => tag.toLowerCase().includes(query.toLowerCase()))
      );
    }
    
    setDisplayedArticles(filtered);
  };



  return (
    <main className="min-h-screen">
      {/* Hero Section */}
      <section className="relative bg-gradient-to-br from-primary-50 via-white to-secondary-50 py-16 sm:py-20 lg:py-24 overflow-hidden">
        {/* Floating Background Elements */}
        <div className="absolute top-20 left-10 w-20 h-20 bg-primary-400/20 rounded-full blur-xl animate-pulse"></div>
        <div className="absolute bottom-20 right-10 w-32 h-32 bg-secondary-400/20 rounded-full blur-xl animate-pulse delay-1000"></div>
        <div className="absolute top-1/2 left-1/4 w-16 h-16 bg-primary-300/20 rounded-full blur-xl animate-pulse delay-500"></div>
        
        <Container>
          <div className="text-center max-w-4xl mx-auto">
            {/* Badge */}
            <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
              <Newspaper className="w-4 h-4" />
              School News & Updates
            </div>

            {/* Title */}
            <h1 className="text-3xl sm:text-4xl lg:text-5xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
              Latest News &
              <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent"> Updates</span>
            </h1>
            
            <p className="text-lg sm:text-xl text-gray-600 leading-relaxed mb-8">
              Stay informed about school events, achievements, announcements, and important updates from our educational community.
            </p>

            {/* Search Bar */}
            <div className="relative max-w-md mx-auto">
              <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search news articles..."
                value={searchQuery}
                onChange={(e) => handleSearch(e.target.value)}
                className="w-full pl-12 pr-4 py-3 rounded-xl border border-gray-200 focus:border-primary-500 focus:ring-2 focus:ring-primary-100 transition-all duration-200"
              />
            </div>
          </div>
        </Container>
      </section>

      {/* News Content */}
      <section className="py-12 sm:py-16 lg:py-20 bg-white">
        <div className='max-w-8xl mx-auto px-4 sm:px-6 lg:px-8'>
          {/* Category Filter */}
          <div className="mb-12">
            <div className="flex items-center gap-2 text-sm text-gray-600 mb-4">
              <Filter className="w-4 h-4" />
              Filter by Category:
            </div>
            <div className="flex flex-wrap gap-3">
              {newsCategories.map((category) => (
                <button
                  key={category}
                  onClick={() => handleCategoryChange(category)}
                  className={`px-4 py-2 rounded-lg font-medium text-sm transition-all duration-200 ${
                    selectedCategory === category
                      ? 'bg-primary-600 text-white shadow-lg'
                      : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                  }`}
                >
                  {category}
                </button>
              ))}
            </div>
          </div>

          {/* Results Count */}
          <div className="mb-8">
            <p className="text-gray-600">
              Showing {displayedArticles.length} article{displayedArticles.length !== 1 ? 's' : ''}
              {selectedCategory !== 'All' && ` in ${selectedCategory}`}
              {searchQuery && ` matching "${searchQuery}"`}
            </p>
          </div>

          {/* Featured Article (First Article) */}
          {displayedArticles.length > 0 && (
            <div className="mb-16">
              <h2 className="text-2xl font-bold text-gray-900 mb-8">Featured Article</h2>
              <FeaturedArticleCard article={displayedArticles[0]} />
            </div>
          )}

          {/* Articles Grid */}
          {displayedArticles.length > 1 ? (
            <div>
              <h2 className="text-2xl font-bold text-gray-900 mb-8">More Articles</h2>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {displayedArticles.slice(1).map((article) => (
                  <ArticleCard key={article.id} article={article} />
                ))}
              </div>
            </div>
          ) : displayedArticles.length === 0 ? (
            <div className="text-center py-16">
              <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                <BookOpen className="w-8 h-8 text-gray-400" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-2">No Articles Found</h3>
              <p className="text-gray-600 mb-6">
                {searchQuery 
                  ? `No articles match your search for "${searchQuery}"`
                  : `No articles available in ${selectedCategory} category`
                }
              </p>
              <button
                onClick={() => {
                  setSelectedCategory('All');
                  setSearchQuery('');
                  setDisplayedArticles(mockNewsData);
                }}
                className="inline-flex items-center gap-2 bg-primary-600 hover:bg-primary-700 text-white px-6 py-3 rounded-lg font-semibold transition-colors duration-200"
              >
                View All Articles
                <ArrowRight className="w-4 h-4" />
              </button>
            </div>
          ) : null}
        </div>
      </section>

      {/* Newsletter Signup */}
      <NewsletterSignup />
    </main>
  );
}

// Featured Article Card Component
function FeaturedArticleCard({ article }: { article: NewsArticle }) {
  return (
    <Card className="overflow-hidden hover:shadow-xl transition-all duration-300 group">
      <div className="grid lg:grid-cols-2 gap-0">
        {/* Image */}
        <div className="relative aspect-[4/3] lg:aspect-auto overflow-hidden">
          <Image
            src={article.featuredImage}
            alt={article.title}
            fill
            className="object-cover group-hover:scale-105 transition-transform duration-500"
            onError={(e) => {
              const target = e.target as HTMLImageElement;
              target.src = `data:image/svg+xml;base64,${btoa(
                `<svg width="400" height="300" xmlns="http://www.w3.org/2000/svg"><rect width="400" height="300" fill="#f3f4f6"/><text x="50%" y="50%" font-family="system-ui" font-size="18" fill="#6b7280" text-anchor="middle" dy=".3em">News Image</text></svg>`
              )}`;
            }}
          />
          <div className="absolute top-4 left-4">
            <Badge variant="secondary">{article.category}</Badge>
          </div>
        </div>

        {/* Content */}
        <div className="p-8 lg:p-12 flex flex-col justify-center">
          <div className="flex items-center gap-4 text-sm text-gray-500 mb-4">
            <div className="flex items-center gap-1">
              <Calendar className="w-4 h-4" />
              {new Date(article.date).toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
              })}
            </div>
            <div className="flex items-center gap-1">
              <Clock className="w-4 h-4" />
              {article.readTime} min read
            </div>
            <div className="flex items-center gap-1">
              <User className="w-4 h-4" />
              {article.author}
            </div>
          </div>

          <h3 className="text-2xl lg:text-3xl font-bold text-gray-900 mb-4 group-hover:text-primary-600 transition-colors duration-300">
            {article.title}
          </h3>

          <p className="text-gray-600 leading-relaxed mb-6 text-lg">
            {article.excerpt}
          </p>

          <div className="flex items-center gap-4">
            <Link
              href={`/news/${article.slug}`}
              className="inline-flex items-center gap-2 bg-primary-600 hover:bg-primary-700 text-white px-6 py-3 rounded-lg font-semibold transition-all duration-200 group-hover:gap-3"
            >
              Read Full Article
              <ArrowRight className="w-4 h-4" />
            </Link>
            
            <div className="flex flex-wrap gap-2">
              {article.tags.slice(0, 3).map((tag) => (
                <Badge key={tag} variant="outline" className="text-xs">
                  {tag}
                </Badge>
              ))}
            </div>
          </div>
        </div>
      </div>
    </Card>
  );
}

// Regular Article Card Component
export function ArticleCard({ article , isLarge = false }: { article: NewsArticle , isLarge?: boolean }) {
  return (
    <Card className={cn("overflow-hidden hover:shadow-xl transition-all duration-300 group hover:scale-105", isLarge ? 'md:col-span-2' : '')}>
      {/* Image */}
      <div className="relative aspect-[4/3] overflow-hidden">
        <Image
          src={article.featuredImage}
          alt={article.title}
          fill
          className="object-cover group-hover:scale-110 transition-transform duration-500"
          onError={(e) => {
            const target = e.target as HTMLImageElement;
            target.src = `data:image/svg+xml;base64,${btoa(
              `<svg width="400" height="300" xmlns="http://www.w3.org/2000/svg"><rect width="400" height="300" fill="#f3f4f6"/><text x="50%" y="50%" font-family="system-ui" font-size="18" fill="#6b7280" text-anchor="middle" dy=".3em">News Image</text></svg>`
            )}`;
          }}
        />
        <div className="absolute top-4 left-4">
          <Badge variant="secondary">{article.category}</Badge>
        </div>
        <div className="absolute inset-0 bg-gradient-to-t from-black/20 via-transparent to-transparent" />
      </div>

      {/* Content */}
      <div className="p-6">
        <div className="flex items-center gap-2 text-sm text-gray-500 mb-3">
          <Calendar className="w-4 h-4" />
          {new Date(article.date).toLocaleDateString('en-US', {
            month: 'short',
            day: 'numeric',
            year: 'numeric'
          })}
          <span>•</span>
          <Clock className="w-4 h-4" />
          {article.readTime} min
        </div>

        <h3 className="text-xl font-bold text-gray-900 mb-3 line-clamp-2 group-hover:text-primary-600 transition-colors duration-300">
          {article.title}
        </h3>

        <p className="text-gray-600 text-sm leading-relaxed mb-4 line-clamp-3">
          {article.excerpt}
        </p>

        <div className="flex items-center justify-between">
          <Link
            href={`/news/${article.slug}`}
            className="inline-flex items-center gap-2 text-primary-600 hover:text-primary-700 font-semibold text-sm group-hover:gap-3 transition-all duration-300"
          >
            Read More
            <ArrowRight className="w-4 h-4" />
          </Link>
          
          <div className="text-xs text-gray-500">
            By {article.author}
          </div>
        </div>
      </div>
    </Card>
  );
}