import React from 'react';
import Card from '@/components/common/ui/card';
import Badge from '@/components/common/ui/badge';
import { 
  User, 
  Users, 
  Heart, 
  BookOpen, 
  Target, 
  Lightbulb,
  Palette,
  MessageCircle,
  Trophy,
  Microscope,
  Monitor,
  Library
} from 'lucide-react';

export function ServicesOverview() {
  const services = [
    {
      id: 1,
      category: "Individual Development",
      title: "Personalized Growth Programs",
      description: "We prioritize personal growth by offering tailored services that help students realize their potential and navigate life's challenges.",
      features: [
        {
          icon: Target,
          title: "Academic and Career Counseling",
          description: "Guide students through crucial academic and career decisions, including course selection, academic planning, and talent identification."
        },
        {
          icon: Heart,
          title: "Psychological Counseling",
          description: "Counseling sessions and workshops to address anxiety, self-esteem challenges, and build resilience and emotional well-being."
        },
        {
          icon: Lightbulb,
          title: "Talent Discovery",
          description: "Opportunities to explore and develop innate abilities across arts, sports, science, and more through tailored programs."
        }
      ],
      color: "from-primary-500 to-primary-600",
      bgColor: "from-primary-50 to-primary-50"
    },
    {
      id: 2,
      category: "Social Development",
      title: "Building Community Connections",
      description: "We believe that social skills and empathy are critical for a well-rounded education. Our programs encourage collaboration, understanding, and community engagement.",
      features: [
        {
          icon: Users,
          title: "Group Programs",
          description: "Foster teamwork through collaborative projects, educational field trips, sports competitions, and seminars."
        },
        {
          icon: Trophy,
          title: "Promoting Responsibility",
          description: "Nurture responsibility, compassion, and commitment to helping others through structured programs."
        },
        {
          icon: Heart,
          title: "Culture of Peace",
          description: "Emphasize peace, empathy, and respect for cultural and religious diversity to become advocates for harmony."
        }
      ],
      color: "from-secondary-500 to-secondary-600",
      bgColor: "from-secondary-50 to-secondary-50"
    },
    {
      id: 3,
      category: "Emotional Development",
      title: "Building Emotional Intelligence",
      description: "Supporting emotional intelligence and self-awareness is at the heart of our mission. We offer programs to enhance communication, confidence, and creativity.",
      features: [
        {
          icon: MessageCircle,
          title: "Communication Skills Training",
          description: "Enhance verbal communication, active listening, and constructive emotional expression for stronger relationships."
        },
        {
          icon: User,
          title: "Boosting Self-Confidence",
          description: "Workshops focused on building self-esteem in a safe and supportive environment that empowers students."
        },
        {
          icon: Palette,
          title: "Encouraging Creativity",
          description: "Support students in exploring creative potential, thinking outside the box, and expressing themselves freely."
        }
      ],
      color: "from-primary-600 to-secondary-600",
      bgColor: "from-primary-50 to-secondary-50"
    },
    {
      id: 4,
      category: "Continuous Learning",
      title: "Lifelong Learning Resources",
      description: "Our educational philosophy prioritizes lifelong learning by offering resources and opportunities for student success.",
      features: [
        {
          icon: BookOpen,
          title: "Supplementary Programs",
          description: "Explore diverse areas like foreign languages, arts, sports, and technology to enrich education."
        },
        {
          icon: Library,
          title: "Library & Educational Resources",
          description: "Diverse, up-to-date resources including books, digital materials, and research tools for academic pursuits."
        },
        {
          icon: Microscope,
          title: "Well-Equipped Laboratories",
          description: "Hands-on learning with well-equipped science and technology labs fostering curiosity and innovation."
        },
        {
          icon: Monitor,
          title: "Computer Lab",
          description: "Top-notch computer lab to learn latest tech, build digital skills, and explore innovations."
        }
      ],
      color: "from-secondary-600 to-primary-600",
      bgColor: "from-secondary-50 to-primary-50"
    }
  ];

  return (
    <section id="services-overview" className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gradient-to-br from-gray-50 via-white to-primary-50/30">
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-20">
          <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
            🎯 Our Core Services
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
            Comprehensive Educational
            <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent"> Services</span>
          </h2>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto leading-relaxed">
            At Setaragan Zirak Private School, our team is dedicated to nurturing each student&apos;s holistic development, 
            going beyond academics to focus on their individual growth, social connections, emotional well-being, and lifelong learning.
          </p>
        </div>

        <div className="space-y-24">
          {services.map((service, index) => (
            <div key={service.id} className="group">
              <Card className={`overflow-hidden shadow-2xl border-0 bg-gradient-to-br ${service.bgColor} hover:shadow-3xl transition-all duration-500 hover:-translate-y-2`}>
                <div className="p-8 md:p-12 lg:p-16">
                  <div className={`flex flex-col ${index % 2 === 0 ? 'lg:flex-row' : 'lg:flex-row-reverse'} gap-12 lg:gap-16`}>
                    
                    {/* Content Section */}
                    <div className="lg:w-3/5 space-y-8">
                      <div className="space-y-6">
                        <Badge className={`bg-gradient-to-r ${service.color} text-white px-4 py-2 text-sm font-semibold`}>
                          {service.category}
                        </Badge>
                        
                        <h3 className="text-2xl md:text-3xl font-bold text-gray-900 leading-tight group-hover:text-primary-600 transition-colors duration-300">
                          {service.title}
                        </h3>
                        
                        <p className="text-lg text-gray-700 leading-relaxed">
                          {service.description}
                        </p>
                      </div>

                      {/* Features List */}
                      <div className="space-y-6">
                        {service.features.map((feature, featureIndex) => (
                          <div key={featureIndex} className="flex gap-4 group/feature">
                            <div className={`flex-shrink-0 w-12 h-12 bg-gradient-to-r ${service.color} rounded-xl flex items-center justify-center shadow-lg group-hover/feature:scale-110 transition-transform duration-300`}>
                              <feature.icon className="w-6 h-6 text-white" />
                            </div>
                            <div className="flex-1 space-y-2">
                              <h4 className="text-xl font-semibold text-gray-900 group-hover/feature:text-primary-600 transition-colors duration-300">
                                {feature.title}
                              </h4>
                              <p className="text-gray-600 leading-relaxed">
                                {feature.description}
                              </p>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>

                    {/* Visual Section */}
                    <div className="lg:w-2/5">
                      <div className="sticky top-8">
                        <div className={`relative aspect-square rounded-3xl bg-gradient-to-br ${service.color} p-8 shadow-2xl overflow-hidden group-hover:scale-105 transition-transform duration-500`}>
                          {/* Background Pattern */}
                          <div className="absolute inset-0 opacity-20">
                            <div className="absolute inset-0" style={{
                              backgroundImage: "url(\"data:image/svg+xml,%3Csvg width='40' height='40' viewBox='0 0 40 40' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='%23ffffff' fill-opacity='0.1'%3E%3Cpath d='M20 20c0-5.5-4.5-10-10-10s-10 4.5-10 10 4.5 10 10 10 10-4.5 10-10zm10 0c0-5.5-4.5-10-10-10s-10 4.5-10 10 4.5 10 10 10 10-4.5 10-10z'/%3E%3C/g%3E%3C/svg%3E\")"
                            }}></div>
                          </div>
                          
                          {/* Content */}
                          <div className="relative z-10 flex flex-col items-center justify-center h-full text-center text-white">
                            <div className="w-20 h-20 bg-white/20 rounded-full flex items-center justify-center mb-6 backdrop-blur-sm">
                              <span className="text-3xl font-bold">{service.id}</span>
                            </div>
                            <h4 className="text-2xl font-bold mb-4">{service.category}</h4>
                            <p className="text-white/90 text-sm leading-relaxed">
                              {service.features.length} specialized programs designed for excellence
                            </p>
                          </div>

                          {/* Floating Elements */}
                          <div className="absolute top-4 right-4 w-8 h-8 bg-white/20 rounded-full animate-pulse"></div>
                          <div className="absolute bottom-6 left-6 w-6 h-6 bg-white/20 rounded-full animate-pulse delay-500"></div>
                          <div className="absolute top-1/3 left-4 w-4 h-4 bg-white/20 rounded-full animate-pulse delay-1000"></div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </Card>
            </div>
          ))}
        </div>

        {/* Commitment Section */}
        <div className="mt-24">
          <Card className="overflow-hidden shadow-2xl border-0 bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 text-white">
            <div className="p-8 md:p-12 lg:p-16 text-center">
              <h3 className="text-2xl md:text-3xl font-bold mb-6">Our Commitment</h3>
              <p className="text-primary-100 text-lg md:text-xl leading-relaxed max-w-4xl mx-auto">
                At Setaragan Zirak Private School, we are committed to shaping well-rounded individuals who are academically competent, 
                socially responsible, emotionally resilient, and eager for lifelong learning. Through our comprehensive programs, 
                we aim to provide students with the tools and confidence they need to succeed in every aspect of their lives.
              </p>
            </div>
          </Card>
        </div>
      </div>
    </section>
  );
}

export default ServicesOverview;
