import React from 'react';
import Button from '@/components/common/ui/button';
import Card from '@/components/common/ui/card';
import { Mail, Phone, MapPin, Users, GraduationCap, Award } from 'lucide-react';
import Link from 'next/link';

export function CallToAction() {
  const opportunities = [
    {
      title: "Teaching Positions",
      description: "Join our dedicated faculty and inspire the next generation of learners",
      icon: GraduationCap,
      action: "View Openings"
    },
    {
      title: "Support Roles",
      description: "Help create an exceptional learning environment for our students",
      icon: Users,
      action: "Explore Roles"
    },
    {
      title: "Leadership Opportunities",
      description: "Lead educational initiatives and drive positive change",
      icon: Award,
      action: "Learn More"
    }
  ];

  return (
    <section id='join-team' className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gradient-to-br from-primary-900 via-primary-800 to-secondary-900 text-white relative overflow-hidden">
      {/* Background Pattern */}
      <div className="absolute inset-0 opacity-10">
        <div className="absolute top-20 left-20 w-32 h-32 bg-white rounded-full blur-3xl animate-pulse"></div>
        <div className="absolute bottom-32 right-16 w-40 h-40 bg-secondary-300 rounded-full blur-3xl animate-pulse delay-1000"></div>
        <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-24 h-24 bg-primary-300 rounded-full blur-2xl animate-pulse delay-500"></div>
      </div>

      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        <div className="text-center mb-16">
          <div className="inline-flex items-center gap-2 bg-white/10 text-white px-4 py-2 rounded-full text-sm font-medium mb-6 backdrop-blur-sm">
            🚀 Join Our Team
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight mb-6">
            Ready to Make a
            <span className="bg-gradient-to-r from-white via-primary-100 to-secondary-100 bg-clip-text text-transparent"> Difference?</span>
          </h2>
          <p className="text-lg text-primary-50 max-w-3xl mx-auto leading-relaxed mb-8">
            Join our exceptional team of educators and professionals committed to shaping the future through quality education. 
            Together, we can create lasting impact in our students&apos; lives.
          </p>
        </div>

        {/* Career Opportunities */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-16">
          {opportunities.map((opportunity, index) => (
            <Card key={index} className="bg-white/10 backdrop-blur-sm border-white/20 text-white p-8 hover:bg-white/15 transition-all duration-300 hover:-translate-y-2 group">
              <div className="text-center space-y-6">
                <div className="w-16 h-16 bg-gradient-to-r from-white/20 to-white/30 rounded-full flex items-center justify-center mx-auto group-hover:scale-110 transition-transform duration-300">
                  <opportunity.icon className="w-8 h-8 text-white" />
                </div>
                <div>
                  <h3 className="text-2xl font-bold mb-3">{opportunity.title}</h3>
                  <p className="text-primary-50 mb-6 leading-relaxed">
                    {opportunity.description}
                  </p>
                </div>
                <Button 
                  variant="outlined" 
                  className="border-white! text-white hover:bg-white hover:text-primary-600! transition-colors duration-300 w-full"
                >
                  {opportunity.action}
                </Button>
              </div>
            </Card>
          ))}
        </div>

        {/* Contact Information */}
        <div className="bg-white/10 backdrop-blur-sm rounded-3xl p-12 border border-white/20">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div>
              <h3 className="text-2xl md:text-3xl font-bold mb-6">
                Get in Touch with Our Team
              </h3>
              <p className="text-primary-50 text-lg mb-8 leading-relaxed">
                Have questions about career opportunities or want to learn more about joining our educational community? 
                We&apos;d love to hear from you.
              </p>
              
              <div className="space-y-4">
                <div className="flex items-center gap-4">
                  <div className="w-12 h-12 bg-white/20 rounded-full flex items-center justify-center">
                    <Mail className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <p className="font-semibold">Email Us</p>
                    <p className="text-primary-50">careers@setaraganzirak.edu.af</p>
                  </div>
                </div>
                
                <div className="flex items-center gap-4">
                  <div className="w-12 h-12 bg-white/20 rounded-full flex items-center justify-center">
                    <Phone className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <p className="font-semibold">Call Us</p>
                    <p className="text-primary-50">+93 (0) 20 123 4567</p>
                  </div>
                </div>
                
                <div className="flex items-center gap-4">
                  <div className="w-12 h-12 bg-white/20 rounded-full flex items-center justify-center">
                    <MapPin className="w-6 h-6 text-white" />
                  </div>
                  <div>
                    <p className="font-semibold">Visit Us</p>
                    <p className="text-primary-50">Karte Naw, Kabul, Afghanistan</p>
                  </div>
                </div>
              </div>
            </div>

            <div className="bg-white/5 rounded-2xl p-8 border border-white/10">
              <h4 className="text-2xl font-bold mb-6 text-center">Quick Application</h4>
              <form className="space-y-4">
                <div>
                  <input
                    type="text"
                    placeholder="Full Name"
                    className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-xl text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300"
                  />
                </div>
                <div>
                  <input
                    type="email"
                    placeholder="Email Address"
                    className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-xl text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300"
                  />
                </div>
                <div>
                  <select className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-xl text-white focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300">
                    <option value="" className="text-gray-900">Position of Interest</option>
                    <option value="teacher" className="text-gray-900">Teaching Position</option>
                    <option value="support" className="text-gray-900">Support Role</option>
                    <option value="admin" className="text-gray-900">Administrative Role</option>
                    <option value="leadership" className="text-gray-900">Leadership Position</option>
                  </select>
                </div>
                <div>
                  <textarea
                    placeholder="Tell us about your experience and interest..."
                    rows={4}
                    className="w-full px-4 py-3 bg-white/10 border border-white/20 rounded-xl text-white placeholder-white/60 focus:outline-none focus:ring-2 focus:ring-white/30 transition-all duration-300 resize-none"
                  ></textarea>
                </div>
                <Button variant='contained' color='dark' className="w-full bg-white text-primary-600! hover:bg-gray-100! transition-colors duration-300 py-3 font-semibold">
                  Submit Application
                </Button>
              </form>
            </div>
          </div>
        </div>

        {/* Final CTA */}
        <div className="text-center mt-16">
          <h3 className="text-2xl md:text-3xl font-bold mb-4">
            Together, We Shape the Future
          </h3>
          <p className="text-primary-50 text-lg mb-8 max-w-2xl mx-auto">
            Join us in our mission to provide exceptional education and create opportunities 
            for every student to reach their full potential.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button variant='contained' color='dark' className="bg-white text-primary-600! hover:bg-gray-100! px-8 py-4 font-semibold">
              View All Positions
            </Button>
            <Link 
              href="/contact"
            >
              <Button
                variant='outlined' color='primary'
                className="border-white! text-white hover:bg-white! hover:text-primary-600! px-8 py-4 font-semibold"
              >
                Schedule a Visit
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </section>
  );
}

export default CallToAction;
