"use client";

import React from "react";
import Card from "@/components/common/ui/card";
import Badge from "@/components/common/ui/badge";
import { Crown, GraduationCap, Users, Heart } from "lucide-react";
import Link from "next/link";
import { useLeadership } from "@/hooks/useLeadership";
import classNames from "classnames";

const iconMap: Record<string, React.FC<{ className?: string }>> = {
  "Founder & Director": Crown,
  Principal: GraduationCap,
  "Communication Coordinator": Users,
  "Vice Principal": Heart,
  "Finance & Communications Manager": Users,
};

export function Leadership() {
  const { data: leaders = [], isLoading, error } = useLeadership();

  if (error) {
    return (
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <p className="text-red-600">Failed to load leadership team</p>
          </div>
        </div>
      </section>
    );
  }

  if (isLoading) {
    return (
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <p className="text-gray-600">Loading leadership team...</p>
          </div>
        </div>
      </section>
    );
  }

  return (
    <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-white">
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-20">
          <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
            👑 Leadership Team
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
            Our School
            <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent">
              {" "}
              Leadership
            </span>
          </h2>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Meet the visionary leaders who guide our school with passion,
            expertise, and an unwavering commitment to educational excellence
            and student success.
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
          {leaders.map((leader) => {
            const IconComponent = iconMap[leader.position] || Users;
            const expertiseArray = Array.isArray(leader.expertise)
              ? leader.expertise
              : typeof leader.expertise === "string"
              ? JSON.parse(leader.expertise)
              : [];

            return (
              <Card
                key={leader.id}
                className={classNames(
                  "overflow-hidden shadow-2xl border-0 bg-gradient-to-br from-white via-white to-primary-50/30 hover:shadow-3xl transition-all duration-500 hover:-translate-y-2 group",
                  {"lg:col-span-2 w-fit mx-auto": leader.position === "Founder & Director"} 
                )}
              >
                <div className={classNames("p-8 lg:p-10",
                  {"lg:max-w-3xl lg:mx-auto": leader.position === "Founder & Director"}
                )}>
                  <div className="flex flex-col md:flex-row gap-8">
                    {/* Photo Section */}
                    <div className="md:w-1/3">
                      <div className="relative">
                        <div className="aspect-[4/5] bg-gradient-to-br from-gray-200 to-gray-300 rounded-2xl overflow-hidden shadow-xl">
                          <div className="w-full h-full bg-gradient-to-br from-primary-500 to-secondary-500 flex items-center justify-center">
                            <IconComponent className="w-16 h-16 text-white opacity-60" />
                          </div>
                        </div>
                        <div className="absolute -bottom-4 -right-4 w-12 h-12 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center shadow-lg group-hover:scale-110 transition-transform duration-300">
                          <IconComponent className="w-6 h-6 text-white" />
                        </div>
                      </div>
                    </div>

                    {/* Content Section */}
                    <div className="md:w-2/3 space-y-6">
                      <div>
                        <h3 className="text-2xl md:text-3xl font-bold text-gray-900 mb-2 group-hover:text-primary-600 transition-colors duration-300">
                          {leader.name}
                        </h3>
                        <Badge
                          variant="secondary"
                          className="bg-gradient-to-r from-primary-500 to-secondary-500 text-white px-4 py-2 text-sm font-semibold"
                        >
                          {leader.position}
                        </Badge>
                      </div>

                      <p className="text-gray-700 leading-relaxed">
                        {leader.bio}
                      </p>

                      {expertiseArray.length > 0 && (
                        <div>
                          <h4 className="font-semibold text-gray-900 mb-3">
                            Areas of Expertise:
                          </h4>
                          <div className="flex flex-wrap gap-2">
                            {expertiseArray.map(
                              (skill: string, skillIndex: number) => (
                                <Badge
                                  key={skillIndex}
                                  variant="outline"
                                  className="border-primary-300 text-primary-700 bg-primary-50 px-3 py-1 text-xs"
                                >
                                  {skill}
                                </Badge>
                              )
                            )}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              </Card>
            );
          })}
        </div>

        {/* Leadership Philosophy */}
        <div className="mt-20 bg-gradient-to-r from-primary-600 to-secondary-600 text-white p-12 rounded-3xl shadow-2xl text-center">
          <h3 className="text-2xl md:text-3xl font-bold mb-6">
            Our Leadership Philosophy
          </h3>
          <p className="text-primary-100 text-lg mb-8 max-w-3xl mx-auto leading-relaxed">
            &ldquo;We believe that great leadership in education comes from a
            deep commitment to student success, innovative teaching methods, and
            creating an environment where every individual can flourish and
            reach their full potential.&rdquo;
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link
              href="#join-team"
              className="bg-white text-primary-600 px-8 py-4 rounded-xl font-semibold hover:bg-gray-50 transition-colors duration-300 shadow-lg"
            >
              Join Our Team
            </Link>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-xl font-semibold hover:bg-white hover:text-primary-600 transition-colors duration-300"
            >
              Contact Leadership
            </Link>
          </div>
        </div>
      </div>
    </section>
  );
}

export default Leadership;
