"use client";

import { SafeImage } from "@/components/common/ui";
import Badge from "@/components/common/ui/badge";
import Card from "@/components/common/ui/card";
import {
  Beaker,
  BookOpen,
  Calculator,
  Eye,
  Globe,
  GraduationCap,
  Palette,
  Users,
} from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import teachersData from "@/constants/teachers.json"; // ✅ Static import
import { Teacher } from "./types";


// ✅ Dictionary mapping for subject icons
const subjectIcons: Record<string, React.ElementType> = {
  math: Calculator,
  physics: Calculator,
  chemistry: Beaker,
  science: Beaker,
  english: Globe,
  language: Globe,
  computer: Palette,
  technology: Palette,
  sociology: Users,
  speaking: Users,
};

const getSubjectIcon = (subject: string) => {
  const key = Object.keys(subjectIcons).find((k) =>
    subject.toLowerCase().includes(k)
  );
  return key ? subjectIcons[key] : BookOpen;
};

// ✅ Truncate helper
const truncateBio = (bio: string, maxLength: number = 150) => {
  if (bio.length <= maxLength) return bio;
  return bio.substring(0, maxLength).trim() + "...";
};

// ✅ Slugify helper for keys and IDs
const slugify = (text: string) =>
  text.toLowerCase().replace(/\s+/g, "-").replace(/[^\w-]+/g, "");

export function TeamMembers() {
  const [showAllMale, setShowAllMale] = useState(false);
  const [showAllFemale, setShowAllFemale] = useState(false);

  const teachers: Teacher[] = teachersData;
  const maleTeachers = teachers.filter((t) => t.gender === "male");
  const femaleTeachers = teachers.filter((t) => t.gender === "female");

  const teachersPerRow = 4;
  const initialRows = 2;
  const initialTeachersToShow = teachersPerRow * initialRows;

  const displayedMaleTeachers = showAllMale
    ? maleTeachers
    : maleTeachers.slice(0, initialTeachersToShow);

  const displayedFemaleTeachers = showAllFemale
    ? femaleTeachers
    : femaleTeachers.slice(0, initialTeachersToShow);

  // ✅ Scroll back when collapsing
  const handleToggle = (gender: "male" | "female") => {
    if (gender === "male") {
      setShowAllMale(!showAllMale);
      if (showAllMale) {
        document.getElementById("male-faculty")?.scrollIntoView({
          behavior: "smooth",
        });
      }
    } else {
      setShowAllFemale(!showAllFemale);
      if (showAllFemale) {
        document.getElementById("female-faculty")?.scrollIntoView({
          behavior: "smooth",
        });
      }
    }
  };

  return (
    <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gray-50">
      <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section Header */}
        <div className="text-center mb-20">
          <div className="inline-flex items-center gap-2 bg-secondary-100 text-secondary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
            👥 Our Faculty Team
          </div>
          <h2 className="text-2xl sm:text-3xl md:text-4xl font-bold leading-tight tracking-tight text-gray-900 mb-6">
            Meet Our
            <span className="bg-gradient-to-r from-primary-600 via-primary-500 to-secondary-600 bg-clip-text text-transparent">
              {" "}
              Dedicated Educators
            </span>
          </h2>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Our passionate team of {teachers.length} educators brings diverse
            expertise and years of experience to create an exceptional learning
            environment for every student.
          </p>
        </div>

        {/* Male Teachers Section */}
        {maleTeachers.length > 0 && (
          <div id="male-faculty" className="mb-16">
            <div className="text-center mb-10">
              <div className="inline-flex items-center gap-3 mb-4">
                <div className="w-12 h-12 bg-gradient-to-r from-blue-500 to-blue-600 rounded-full flex items-center justify-center shadow-lg">
                  <Users className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-2xl md:text-3xl font-bold text-gray-900">
                  Male Faculty
                </h3>
              </div>
              <p className="text-lg text-gray-600">
                {maleTeachers.length} dedicated male educators
              </p>
            </div>

            {/* Male Teacher Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
              {displayedMaleTeachers.map((teacher) => {
                const SubjectIcon = getSubjectIcon(teacher.subject);
                const slug = slugify(teacher.name);

                return (
                  <Card
                    id={slug}
                    key={slug}
                    className="overflow-hidden shadow-xl border-0 bg-white hover:shadow-2xl transition-all duration-500 hover:-translate-y-2 group"
                  >
                    <div className="p-1.5 py-3">
                      {/* Teacher Photo */}
                      <div className="relative mb-6">
                        <div className="aspect-square rounded-2xl overflow-hidden shadow-lg bg-gray-200">
                          <SafeImage
                            src={`/teachers_data/${teacher.image.replace(
                              "team_images/",
                              ""
                            )}`}
                            alt={teacher.name}
                            width={300}
                            height={300}
                            className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-105"
                          />
                        </div>
                        <div className="absolute -bottom-3 -right-3 w-10 h-10 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center shadow-lg group-hover:scale-110 transition-transform duration-300">
                          <SubjectIcon className="w-5 h-5 text-white" />
                        </div>
                      </div>

                      {/* Teacher Info */}
                      <div className="space-y-4">
                        <div>
                          <h4 className="text-xl font-bold text-gray-900 mb-1 group-hover:text-primary-600 transition-colors duration-300">
                            {teacher.name}
                          </h4>
                          <p className="text-gray-600 font-medium text-sm">
                            {teacher.subject}
                          </p>
                        </div>

                        <div className="space-y-2">
                          <div className="flex items-center justify-between">
                            <span className="text-sm text-gray-500">
                              Experience:
                            </span>
                            <Badge
                              variant="outline"
                              className="border-primary-300 text-primary-700 bg-primary-50 text-xs"
                            >
                              {teacher.experience}
                            </Badge>
                          </div>
                          <div className="text-sm text-gray-600">
                            <span className="font-medium">Achievement:</span>
                            <div className="mt-1">
                              <Badge className="bg-gradient-to-r from-secondary-500 to-secondary-600 text-white text-xs">
                                {teacher.achievement}
                              </Badge>
                            </div>
                          </div>
                        </div>

                        {/* Bio Section */}
                        {teacher.bio && (
                          <div className="text-sm text-gray-600">
                            <p className="leading-relaxed">
                              {truncateBio(teacher.bio)}
                            </p>
                            {teacher.bio.length > 150 && (
                              <Link
                                href={`/team/${slug}`}
                                className="text-primary-600 hover:text-primary-700 font-medium text-xs mt-2 flex items-center gap-1 transition-colors duration-200 hover:underline"
                              >
                                <Eye className="w-3 h-3" />
                                Read Full Bio
                              </Link>
                            )}
                          </div>
                        )}
                      </div>
                    </div>
                  </Card>
                );
              })}
            </div>

            {/* See More Button */}
            {maleTeachers.length > initialTeachersToShow && (
              <div className="text-center mt-8">
                <button
                  onClick={() => handleToggle("male")}
                  className="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-primary-600 to-secondary-600 hover:from-primary-700 hover:to-secondary-700 text-white font-medium rounded-full shadow-lg hover:shadow-xl transition-all duration-300 focus:outline-none focus:ring-4 focus:ring-primary-200"
                >
                  {showAllMale ? (
                    <>
                      <Eye className="w-4 h-4" />
                      Show Less
                    </>
                  ) : (
                    <>
                      <Users className="w-4 h-4" />
                      See More ({maleTeachers.length - initialTeachersToShow} more)
                    </>
                  )}
                </button>
              </div>
            )}
          </div>
        )}

        {/* Female Teachers Section */}
        {femaleTeachers.length > 0 && (
          <div id="female-faculty" className="mb-16">
            <div className="text-center mb-10">
              <div className="inline-flex items-center gap-3 mb-4">
                <div className="w-12 h-12 bg-gradient-to-r from-pink-500 to-rose-600 rounded-full flex items-center justify-center shadow-lg">
                  <GraduationCap className="w-6 h-6 text-white" />
                </div>
                <h3 className="text-2xl md:text-3xl font-bold text-gray-900">
                  Female Faculty
                </h3>
              </div>
              <p className="text-lg text-gray-600">
                {femaleTeachers.length} dedicated female educators
              </p>
            </div>

            {/* Female Teacher Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
              {displayedFemaleTeachers.map((teacher) => {
                const SubjectIcon = getSubjectIcon(teacher.subject);
                const slug = slugify(teacher.name);

                return (
                  <Card
                    key={slug}
                    className="overflow-hidden shadow-xl border-0 bg-white hover:shadow-2xl transition-all duration-500 hover:-translate-y-2 group"
                  >
                    <div className="p-1.5 py-3">
                      {/* Teacher Photo */}
                      <div className="relative mb-6">
                        <div className="aspect-square rounded-2xl overflow-hidden shadow-lg bg-gray-200">
                          <SafeImage
                            src={`/teachers_data/${teacher.image.replace(
                              "team_images/",
                              ""
                            )}`}
                            alt={teacher.name}
                            width={300}
                            height={300}
                            className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-105"
                          />
                        </div>
                        <div className="absolute -bottom-3 -right-3 w-10 h-10 bg-gradient-to-r from-pink-500 to-rose-500 rounded-full flex items-center justify-center shadow-lg group-hover:scale-110 transition-transform duration-300">
                          <SubjectIcon className="w-5 h-5 text-white" />
                        </div>
                      </div>

                      {/* Teacher Info */}
                      <div className="space-y-4">
                        <div>
                          <h4 className="text-xl font-bold text-gray-900 mb-1 group-hover:text-primary-600 transition-colors duration-300">
                            {teacher.name}
                          </h4>
                          <p className="text-gray-600 font-medium text-sm">
                            {teacher.subject}
                          </p>
                        </div>

                        <div className="space-y-2">
                          <div className="flex items-center justify-between">
                            <span className="text-sm text-gray-500">
                              Experience:
                            </span>
                            <Badge
                              variant="outline"
                              className="border-primary-300 text-primary-700 bg-primary-50 text-xs"
                            >
                              {teacher.experience}
                            </Badge>
                          </div>
                          <div className="text-sm text-gray-600">
                            <span className="font-medium">Achievement:</span>
                            <div className="mt-1">
                              <Badge className="bg-gradient-to-r from-secondary-500 to-secondary-600 text-white text-xs">
                                {teacher.achievement}
                              </Badge>
                            </div>
                          </div>
                        </div>

                        {/* Bio Section */}
                        {teacher.bio && (
                          <div className="text-sm text-gray-600">
                            <p className="leading-relaxed">
                              {truncateBio(teacher.bio)}
                            </p>
                            {teacher.bio.length > 150 && (
                              <Link
                                href={`/team/${slug}`}
                                className="text-primary-600 hover:text-primary-700 font-medium text-xs mt-2 flex items-center gap-1 transition-colors duration-200 hover:underline"
                              >
                                <Eye className="w-3 h-3" />
                                Read Full Bio
                              </Link>
                            )}
                          </div>
                        )}
                      </div>
                    </div>
                  </Card>
                );
              })}
            </div>

            {/* See More Button */}
            {femaleTeachers.length > initialTeachersToShow && (
              <div className="text-center mt-8">
                <button
                  onClick={() => handleToggle("female")}
                  className="inline-flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-pink-500 to-rose-600 hover:from-pink-600 hover:to-rose-700 text-white font-medium rounded-full shadow-lg hover:shadow-xl transition-all duration-300 focus:outline-none focus:ring-4 focus:ring-pink-200"
                >
                  {showAllFemale ? (
                    <>
                      <Eye className="w-4 h-4" />
                      Show Less
                    </>
                  ) : (
                    <>
                      <GraduationCap className="w-4 h-4" />
                      See More ({femaleTeachers.length - initialTeachersToShow} more)
                    </>
                  )}
                </button>
              </div>
            )}
          </div>
        )}

        {/* Team Stats */}
        <div className="mt-20 bg-white p-8 sm:p-12 rounded-3xl shadow-2xl">
          <h3 className="text-2xl md:text-3xl font-bold text-center text-gray-900 mb-12">
            Our Faculty by the Numbers
          </h3>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
            <div className="text-center">
              <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-2">
                {teachers.length}
              </div>
              <p className="text-gray-600 font-medium">Total Faculty</p>
            </div>
            <div className="text-center">
              <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-2">
                {maleTeachers.length}
              </div>
              <p className="text-gray-600 font-medium">Male Teachers</p>
            </div>
            <div className="text-center">
              <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-2">
                {femaleTeachers.length}
              </div>
              <p className="text-gray-600 font-medium">Female Teachers</p>
            </div>
            <div className="text-center">
              <div className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent mb-2">
                100%
              </div>
              <p className="text-gray-600 font-medium">
                Committed to Excellence
              </p>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}

export default TeamMembers;
