"use client";

import Badge from "@/components/common/ui/badge";
import Card from "@/components/common/ui/card";
import TeacherImage from "@/components/pages/team/TeacherImage";
import teachersData from "@/constants/teachers.json"; // ✅ Static import
import {
  ArrowLeft,
  Award,
  BookOpen,
  Clock,
  GraduationCap,
  Heart,
  Mail,
  Star,
  User,
  Users,
} from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import { Teacher } from "./types";

export default function TeamMember({ teacher }: { teacher: Teacher }) {
  const [teachers, setTeachers] = useState<Teacher[]>(teachersData);

  return (
    <>
      {/* Hero Section */}
      <section className="relative bg-gradient-to-br from-primary-50 via-white to-secondary-50 py-8 sm:py-12 md:py-16 lg:py-20 overflow-hidden">
        {/* Floating Background Elements */}
        <div className="absolute top-20 left-10 w-20 h-20 bg-primary-400/20 rounded-full blur-xl animate-pulse"></div>
        <div className="absolute bottom-20 right-10 w-32 h-32 bg-secondary-400/20 rounded-full blur-xl animate-pulse delay-1000"></div>
        <div className="absolute top-1/2 left-1/4 w-16 h-16 bg-primary-300/20 rounded-full blur-xl animate-pulse delay-500"></div>

        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Back Navigation */}
          <div className="mb-8 lg:mb-12">
            <Link
              href={`/team#${teacher.name.toLowerCase().replace(/\s+/g, "-")}`}
              className="inline-flex items-center gap-2 text-primary-600 hover:text-primary-700 font-medium transition-colors duration-200 group"
            >
              <ArrowLeft className="w-4 h-4 group-hover:-translate-x-1 transition-transform duration-200" />
              Back to Team
            </Link>
          </div>

          <div className="grid lg:grid-cols-2 gap-8 lg:gap-16 items-center">
            {/* Left Column - Teacher Info */}
            <div className="text-center lg:text-left">
              {/* Badge */}
              <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full text-sm font-medium mb-6">
                <User className="w-4 h-4" />
                Faculty Member
              </div>

              {/* Teacher Name & Title */}
              <h1 className="text-3xl sm:text-4xl lg:text-5xl font-bold leading-tight tracking-tight text-gray-900 mb-4">
                {teacher.name}
              </h1>

              <p className="text-xl sm:text-2xl text-primary-600 font-semibold mb-6">
                {teacher.subject}
                <span className="bg-gradient-to-r from-primary-600 to-secondary-600 bg-clip-text text-transparent">
                  {" "}
                  Teacher
                </span>
              </p>

              {/* Experience & Achievement */}
              <div className="flex flex-wrap gap-3 justify-center lg:justify-start mb-8">
                <Badge variant="secondary" className="flex items-center gap-2">
                  <Clock className="w-4 h-4" />
                  {teacher.experience}
                </Badge>
                <Badge variant="outline" className="flex items-center gap-2">
                  <Award className="w-4 h-4" />
                  Top Performer
                </Badge>
              </div>

              {/* Achievement Highlight Card */}
              <Card className="p-6 bg-white/70 backdrop-blur-sm border border-gray-100 shadow-lg mb-8">
                <div className="flex items-center gap-3 mb-3">
                  <div className="w-10 h-10 bg-gradient-to-r from-primary-500 to-secondary-500 rounded-full flex items-center justify-center">
                    <Star className="w-5 h-5 text-white" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    Key Achievement
                  </h3>
                </div>
                <p className="text-gray-700 leading-relaxed">
                  {teacher.achievement}
                </p>
              </Card>

              {/* Action Buttons */}
              <div className="flex flex-col sm:flex-row gap-4 justify-center lg:justify-start">
                <Link
                  href="/contact"
                  className="inline-flex items-center justify-center gap-2 bg-primary-600 hover:bg-primary-700 text-white px-6 py-3 rounded-lg font-semibold transition-all duration-200 shadow-lg hover:shadow-xl"
                >
                  <Mail className="w-5 h-5" />
                  Contact School
                </Link>
                <Link
                  href="/admissions"
                  className="inline-flex items-center justify-center gap-2 border-2 border-primary-300 text-primary-600 px-6 py-3 rounded-lg font-semibold hover:bg-primary-50 hover:border-primary-400 transition-all duration-200"
                >
                  <GraduationCap className="w-5 h-5" />
                  Learn More
                </Link>
              </div>
            </div>

            {/* Right Column - Teacher Photo */}
            <div className="flex justify-center lg:justify-end">
              <div className="relative">
                {/* Main Photo */}
                <div className="relative w-80 h-80 lg:w-96 lg:h-96 rounded-2xl overflow-hidden shadow-2xl bg-white">
                  <TeacherImage
                    src={`/teachers_data/${teacher.image.replace(
                      "team_images/",
                      ""
                    )}`}
                    alt={teacher.name}
                    fill
                    className="object-cover"
                    priority
                  />
                </div>

                {/* Decorative Elements */}
                <div className="absolute -top-4 -right-4 w-24 h-24 bg-gradient-to-br from-primary-400 to-secondary-400 rounded-full opacity-20 animate-pulse"></div>
                <div className="absolute -bottom-4 -left-4 w-16 h-16 bg-gradient-to-br from-secondary-400 to-primary-400 rounded-full opacity-30 animate-bounce"></div>

                {/* Subject Icon Badge */}
                <div className="absolute top-4 right-4 w-12 h-12 bg-white/90 backdrop-blur-sm rounded-full flex items-center justify-center shadow-lg">
                  <BookOpen className="w-6 h-6 text-primary-600" />
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Content Section */}
      <section className="py-8 sm:py-12 md:py-14 lg:py-16 xl:py-18 bg-gray-50">
        <div className="max-w-8xl mx-auto px-4 sm:px-6 lg:px-8">
          {/* Biography Section */}
          <div className="grid lg:grid-cols-3 gap-8">
            {/* Main Biography */}
            <div className="lg:col-span-2">
              <Card className="p-6 sm:p-8">
                <div className="flex items-center gap-3 mb-6">
                  <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                    <User className="w-5 h-5 text-blue-600" />
                  </div>
                  <h2 className="text-2xl font-bold text-gray-900">
                    Biography
                  </h2>
                </div>

                <div className="prose prose-lg max-w-none text-gray-700">
                  {teacher.bio
                    .split("\n")
                    .filter((p) => p.trim())
                    .map((paragraph, index) => (
                      <p key={index} className="mb-4 leading-relaxed">
                        {paragraph.trim()}
                      </p>
                    ))}
                </div>
              </Card>
            </div>

            {/* Sidebar Info */}
            <div className="space-y-6">
              {/* Experience Card */}
              <Card className="p-6">
                <div className="flex items-center gap-3 mb-4">
                  <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                    <GraduationCap className="w-5 h-5 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    Experience
                  </h3>
                </div>
                <p className="text-gray-700 font-medium">
                  {teacher.experience}
                </p>
              </Card>

              {/* Subject Specialization */}
              <Card className="p-6">
                <div className="flex items-center gap-3 mb-4">
                  <div className="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center">
                    <BookOpen className="w-5 h-5 text-purple-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    Specialization
                  </h3>
                </div>
                <Badge variant="secondary" className="text-sm">
                  {teacher.subject}
                </Badge>
              </Card>

              {/* School Values */}
              <Card className="p-6 bg-gradient-to-br from-blue-50 to-purple-50 border-blue-200">
                <div className="flex items-center gap-3 mb-4">
                  <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                    <Heart className="w-5 h-5 text-blue-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">
                    Our Values
                  </h3>
                </div>
                <ul className="space-y-2 text-sm text-gray-700">
                  <li className="flex items-center gap-2">
                    <Star className="w-4 h-4 text-yellow-500" />
                    Excellence in Education
                  </li>
                  <li className="flex items-center gap-2">
                    <Users className="w-4 h-4 text-blue-500" />
                    Community Building
                  </li>
                  <li className="flex items-center gap-2">
                    <Heart className="w-4 h-4 text-red-500" />
                    Compassionate Teaching
                  </li>
                </ul>
              </Card>

              {/* Call to Action */}
              <Card className="p-6 text-center bg-gradient-to-r from-blue-600 to-purple-600 text-white border-0">
                <h3 className="text-lg font-semibold mb-3">
                  Interested in Our School?
                </h3>
                <p className="text-blue-100 text-sm mb-4">
                  Learn more about our programs and how to enroll your child.
                </p>
                <Link
                  href="/admissions"
                  className="inline-block bg-white text-blue-600 px-4 py-2 rounded-lg font-semibold text-sm hover:bg-blue-50 transition-colors duration-200"
                >
                  View Admissions
                </Link>
              </Card>
            </div>
          </div>

          {/* Related Teachers Section */}
          <div className="mt-16">
            <h2 className="text-2xl font-bold text-gray-900 mb-8 text-center">
              Meet Other Team Members
            </h2>
            <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-6">
              {teachers
                .filter((t) => t.name !== teacher.name)
                .slice(0, 4)
                .map((relatedTeacher) => (
                  <Link
                    key={relatedTeacher.name}
                    href={`/team/${relatedTeacher.name
                      .toLowerCase()
                      .replace(/\s+/g, "-")}`}
                    className="group"
                  >
                    <Card className="p-4 text-center hover:shadow-lg transition-all duration-200 group-hover:scale-105">
                      <div className="w-20 h-20 rounded-full overflow-hidden mx-auto mb-3 bg-gray-100">
                        <TeacherImage
                          src={`/teachers_data/${relatedTeacher.image.replace(
                            "team_images/",
                            ""
                          )}`}
                          alt={relatedTeacher.name}
                          width={80}
                          height={80}
                          className="w-full h-full object-cover"
                        />
                      </div>
                      <h3 className="font-semibold text-gray-900 text-sm mb-1">
                        {relatedTeacher.name}
                      </h3>
                      <p className="text-gray-600 text-xs">
                        {relatedTeacher.subject}
                      </p>
                    </Card>
                  </Link>
                ))}
            </div>
          </div>
        </div>
      </section>
    </>
  );
}
