"use client";

import React, { createContext, useContext, ReactNode } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";

interface NewsArticle {
  id: string;
  title: string;
  content: string;
  excerpt: string;
  status: "draft" | "published";
  authorId: string;
  createdAt: Date;
  updatedAt: Date;
}

interface NewsContextType {
  // Queries
  articles: NewsArticle[] | undefined;
  isLoading: boolean;
  error: Error | null;
  
  // Mutations
  createArticle: (data: Partial<NewsArticle>) => Promise<NewsArticle>;
  updateArticle: (id: string, data: Partial<NewsArticle>) => Promise<NewsArticle>;
  deleteArticle: (id: string) => Promise<void>;
  
  // State
  page: number;
  setPage: (page: number) => void;
  search: string;
  setSearch: (search: string) => void;
}

const NewsContext = createContext<NewsContextType | undefined>(undefined);

export function NewsProvider({ children }: { children: ReactNode }) {
  const queryClient = useQueryClient();
  const [page, setPage] = React.useState(1);
  const [search, setSearch] = React.useState("");

  // Fetch articles
  const { data, isLoading, error } = useQuery({
    queryKey: ["news", page, search],
    queryFn: async () => {
      const params = new URLSearchParams({
        page: page.toString(),
        limit: "10",
        search,
      });
      const res = await fetch(`/api/admin/news?${params}`);
      if (!res.ok) throw new Error("Failed to fetch articles");
      return res.json();
    },
  });

  // Create mutation
  const createMutation = useMutation({
    mutationFn: async (newArticle: Partial<NewsArticle>) => {
      const res = await fetch("/api/admin/news", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(newArticle),
      });
      if (!res.ok) throw new Error("Failed to create article");
      const { article } = await res.json();
      return article;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });

  // Update mutation
  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: string; data: Partial<NewsArticle> }) => {
      const res = await fetch(`/api/admin/news/${id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!res.ok) throw new Error("Failed to update article");
      const { article } = await res.json();
      return article;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: string) => {
      const res = await fetch(`/api/admin/news/${id}`, {
        method: "DELETE",
      });
      if (!res.ok) throw new Error("Failed to delete article");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });

  const value: NewsContextType = {
    articles: data?.articles || [],
    isLoading,
    error: error as Error | null,
    createArticle: (data) => createMutation.mutateAsync(data),
    updateArticle: (id, data) => updateMutation.mutateAsync({ id, data }),
    deleteArticle: (id) => deleteMutation.mutateAsync(id),
    page,
    setPage,
    search,
    setSearch,
  };

  return <NewsContext.Provider value={value}>{children}</NewsContext.Provider>;
}

export function useNews() {
  const context = useContext(NewsContext);
  if (!context) {
    throw new Error("useNews must be used within NewsProvider");
  }
  return context;
}
