import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import type { UserPagePermissions } from "@/lib/admin/page-permissions";

export interface AdminUser {
  id: string;
  name: string;
  email: string;
  role: "admin" | "editor" | "viewer";
  status: "pending" | "approved" | "rejected" | "suspended";
  active: boolean;
  permissions?: UserPagePermissions | null; // JSON field for custom page-level permissions
  lastLogin?: string;
  createdAt: string;
  approvalDate?: string;
  rejectionReason?: string;
}

export interface UsersListResponse {
  users: AdminUser[];
  pagination: {
    total: number;
    limit: number;
    offset: number;
    pages: number;
  };
}

interface FetchUsersParams {
  status?: string;
  role?: string;
  search?: string;
  limit?: number;
  offset?: number;
}

/**
 * Fetch admin users with optional filtering
 */
const fetchUsers = async (params: FetchUsersParams): Promise<UsersListResponse> => {
  const searchParams = new URLSearchParams();

  if (params.status) searchParams.append("status", params.status);
  if (params.role) searchParams.append("role", params.role);
  if (params.search) searchParams.append("search", params.search);
  if (params.limit) searchParams.append("limit", params.limit.toString());
  if (params.offset) searchParams.append("offset", params.offset.toString());

  const response = await fetch(`/api/admin/users?${searchParams.toString()}`);
  if (!response.ok) throw new Error("Failed to fetch users");
  const data = await response.json();
  return data.data;
};

/**
 * Hook to fetch admin users with filters
 */
export const useAdminUsers = (params: FetchUsersParams = {}) => {
  return useQuery({
    queryKey: ["admin-users", params],
    queryFn: () => fetchUsers(params),
  });
};

/**
 * Hook to approve a pending user
 */
export const useApproveUser = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ 
      userId, 
      role = 'viewer',
      permissions = null 
    }: { 
      userId: string; 
      role?: 'admin' | 'editor' | 'viewer';
      permissions?: UserPagePermissions | null;
    }) => {
      const response = await fetch(`/api/admin/users/${userId}/approve`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ role, permissions }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to approve user");
      }

      return response.json();
    },
    onSuccess: () => {
      // Invalidate users list to refetch
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    },
  });
};

/**
 * Hook to reject a pending user
 */
export const useRejectUser = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ userId, reason }: { userId: string; reason?: string }) => {
      const response = await fetch(`/api/admin/users/${userId}/reject`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ reason }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to reject user");
      }

      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    },
  });
};

/**
 * Hook to suspend an approved user
 */
export const useSuspendUser = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ userId, reason }: { userId: string; reason?: string }) => {
      const response = await fetch(`/api/admin/users/${userId}/suspend`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ reason }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to suspend user");
      }

      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    },
  });
};

/**
 * Hook to delete a user
 */
export const useDeleteUser = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (userId: string) => {
      const response = await fetch(`/api/admin/users/${userId}/delete`, {
        method: "DELETE",
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to delete user");
      }

      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    },
  });
};

/**
 * Hook to update user role and/or permissions
 * @param userId - User ID
 * @param role - New role (optional)
 * @param permissions - Custom permissions object (optional)
 */
export const useUpdateUserPermissions = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ 
      userId, 
      role, 
      permissions 
    }: { 
      userId: string; 
      role?: 'admin' | 'editor' | 'viewer';
      permissions?: UserPagePermissions | null;
    }) => {
      const response = await fetch(`/api/admin/users/${userId}/update`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ role, permissions }),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to update user permissions");
      }

      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    },
  });
};

