import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';

interface AdmissionsInquiry {
  id: string;
  studentName: string;
  gradeLevel: string;
  parentName: string;
  parentEmail: string;
  parentPhone: string;
  message?: string;
  hearAbout?: string;
  status: 'new' | 'contacted' | 'enrolled' | 'rejected';
  reviewed: boolean;
  createdAt: string;
}

interface AdmissionsResponse {
  admissions: AdmissionsInquiry[];
}

interface UpdateAdmissionPayload {
  id: string;
  status: string;
}

interface ApiResponse {
  success: boolean;
  message: string;
}

// Fetch all admissions
const fetchAdmissions = async (): Promise<AdmissionsResponse> => {
  const response = await fetch('/api/admin/admissions');
  if (!response.ok) throw new Error('Failed to fetch admissions');
  const result = await response.json();
  return { admissions: result.data };
};

// Delete an admission
const deleteAdmissionApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/admissions/${id}`, { method: 'DELETE' });
  if (!response.ok) throw new Error('Failed to delete admission');
  return response.json();
};

// Update admission status
const updateAdmissionStatusApi = async ({
  id,
  status,
}: UpdateAdmissionPayload): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/admissions/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ status }),
  });
  if (!response.ok) throw new Error('Failed to update admission');
  return response.json();
};

/**
 * Hook to fetch all admissions inquiries
 */
export const useAdmissions = () => {
  return useQuery({
    queryKey: ['admissions'],
    queryFn: fetchAdmissions,
  });
};

/**
 * Hook to delete an admission inquiry
 */
export const useDeleteAdmission = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: deleteAdmissionApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['admissions'] });
    },
  });
};

/**
 * Hook to update admission status
 */
export const useUpdateAdmissionStatus = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: updateAdmissionStatusApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['admissions'] });
    },
  });
};
