import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';

interface ContactMessage {
  id: string;
  name: string;
  email: string;
  phone?: string;
  subject: string;
  message: string;
  read: boolean;
  replied: boolean;
  replyMessage?: string;
  repliedAt?: string;
  createdAt: string;
}

interface ContactsResponse {
  contacts: ContactMessage[];
}

interface ReplyPayload {
  id: string;
  message: string;
}

interface ApiResponse {
  success: boolean;
  message: string;
}

// Fetch all contacts
const fetchContacts = async (): Promise<ContactsResponse> => {
  const response = await fetch('/api/admin/contacts');
  if (!response.ok) throw new Error('Failed to fetch contacts');
  const result = await response.json();
  return { contacts: result.data };
};

// Delete a contact
const deleteContactApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/contacts/${id}`, { method: 'DELETE' });
  if (!response.ok) throw new Error('Failed to delete contact');
  return response.json();
};

// Mark as read
const markAsReadApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/contacts/${id}/read`, { method: 'PUT' });
  if (!response.ok) throw new Error('Failed to mark as read');
  return response.json();
};

// Send reply
const sendReplyApi = async ({ id, message }: ReplyPayload): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/contacts/${id}/reply`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ message }),
  });
  if (!response.ok) throw new Error('Failed to send reply');
  return response.json();
};

/**
 * Hook to fetch all contacts
 */
export const useContacts = () => {
  return useQuery({
    queryKey: ['contacts'],
    queryFn: fetchContacts,
  });
};

/**
 * Hook to delete a contact
 */
export const useDeleteContact = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: deleteContactApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['contacts'] });
    },
  });
};

/**
 * Hook to mark contact as read
 */
export const useMarkContactAsRead = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: markAsReadApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['contacts'] });
    },
  });
};

/**
 * Hook to send a reply to a contact
 */
export const useSendContactReply = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: sendReplyApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['contacts'] });
    },
  });
};
