'use client';

import { useEffect, useState } from 'react';

/**
 * Hook to fetch and manage CSRF tokens for auth forms
 * Usage:
 * const { token, loading, error } = useCsrfToken();
 * Then include token in form submissions
 */
export function useCsrfToken() {
  const [token, setToken] = useState<string | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchCsrfToken = async () => {
      try {
        setLoading(true);
        const response = await fetch('/api/auth/csrf-token');
        
        if (!response.ok) {
          throw new Error('Failed to fetch CSRF token');
        }

        const data = await response.json();
        setToken(data.token);
        setError(null);
      } catch (err) {
        console.error('CSRF token fetch error:', err);
        setError(err instanceof Error ? err.message : 'Unknown error');
        setToken(null);
      } finally {
        setLoading(false);
      }
    };

    fetchCsrfToken();
  }, []);

  return { token, loading, error };
}

/**
 * Hook to add CSRF token to form data
 * Usage:
 * const withCsrf = useAddCsrfToForm();
 * const data = withCsrf({ email, password });
 */
export function useAddCsrfToForm() {
  return (data: Record<string, unknown>, token?: string | null) => {
    if (token) {
      return {
        ...data,
        _csrf: token,
      };
    }
    return data;
  };
}
