import { useEffect, useState } from "react";

/**
 * Custom hook to debounce a value
 * Delays updating the value until after the specified delay has passed without changes
 * @param value - The value to debounce
 * @param delay - Delay in milliseconds (default: 500ms)
 * @returns The debounced value
 */
export function useDebounce<T>(value: T, delay: number = 500): T {
  const [debouncedValue, setDebouncedValue] = useState<T>(value);

  useEffect(() => {
    // Set up a timer to update the debounced value
    const handler = setTimeout(() => {
      setDebouncedValue(value);
    }, delay);

    // Clear the timer if the value changes before the delay is up
    return () => clearTimeout(handler);
  }, [value, delay]);

  return debouncedValue;
}
