import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';

interface Donation {
  id: string;
  name: string;
  email: string;
  phone?: string;
  amount: number;
  currency: string;
  category?: string;
  message?: string;
  paymentMethod?: string;
  anonymous: boolean;
  paymentStatus: 'pending' | 'completed' | 'failed';
  transactionId?: string;
  createdAt: string;
}

interface DonationsResponse {
  donations: Donation[];
}

interface ApiResponse {
  success: boolean;
  message: string;
}

// Fetch all donations
const fetchDonations = async (): Promise<DonationsResponse> => {
  const response = await fetch('/api/admin/donations');
  if (!response.ok) throw new Error('Failed to fetch donations');
  const result = await response.json();
  return { donations: result.data };
};

// Delete a donation
const deleteDonationApi = async (id: string): Promise<ApiResponse> => {
  const response = await fetch(`/api/admin/donations/${id}`, { method: 'DELETE' });
  if (!response.ok) throw new Error('Failed to delete donation');
  return response.json();
};

// Create a manual donation
interface CreateManualDonationPayload {
  name: string;
  email: string;
  phone?: string;
  country?: string;
  amount: number;
  currency: string;
  category?: string;
  paymentMethod: string;
  paymentStatus: 'pending' | 'completed' | 'failed';
  anonymous: boolean;
  message?: string;
  transactionId?: string;
}

const createManualDonationApi = async (
  payload: CreateManualDonationPayload
): Promise<ApiResponse & { donation?: Donation }> => {
  const response = await fetch('/api/admin/donations/add-manual', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(payload),
  });
  if (!response.ok) throw new Error('Failed to create donation');
  return response.json();
};

/**
 * Hook to fetch all donations
 */
export const useDonations = () => {
  return useQuery({
    queryKey: ['donations'],
    queryFn: fetchDonations,
  });
};

/**
 * Hook to delete a donation record
 */
export const useDeleteDonation = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: deleteDonationApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['donations'] });
    },
  });
};

/**
 * Hook to create a manual donation entry
 */
export const useCreateManualDonation = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: createManualDonationApi,
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['donations'] });
    },
  });
};
