import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';

export interface HeroSlide {
  id: string;
  title: string;
  description?: string;
  imageUrl: string;
  alt: string;
  order: number;
  active: boolean;
  createdAt: Date;
  updatedAt: Date;
}

export interface HeroSlideInput {
  title: string;
  description?: string;
  imageUrl: string;
  alt: string;
  order?: number;
  active?: boolean;
}

// Fetch all hero slides (admin)
export function useHeroSlides() {
  return useQuery({
    queryKey: ['hero-slides'],
    queryFn: async () => {
      const response = await fetch('/api/admin/hero-slides');
      if (!response.ok) throw new Error('Failed to fetch hero slides');
      const data = await response.json();
      return data.data as HeroSlide[];
    },
  });
}

// Create hero slide
export function useCreateHeroSlide() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (slideData: HeroSlideInput) => {
      const response = await fetch('/api/admin/hero-slides', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(slideData),
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || 'Failed to create hero slide');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['hero-slides'] });
    },
  });
}

// Update hero slide
export function useUpdateHeroSlide() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async ({ id, ...slideData }: Partial<HeroSlideInput> & { id: string }) => {
      const response = await fetch(`/api/admin/hero-slides/${id}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(slideData),
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || 'Failed to update hero slide');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['hero-slides'] });
    },
  });
}

// Delete hero slide
export function useDeleteHeroSlide() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (id: string) => {
      const response = await fetch(`/api/admin/hero-slides/${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || 'Failed to delete hero slide');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['hero-slides'] });
    },
  });
}

// Fetch public hero slides
export function usePublicHeroSlides() {
  return useQuery({
    queryKey: ['public-hero-slides'],
    queryFn: async () => {
      const response = await fetch('/api/hero-slides');
      if (!response.ok) throw new Error('Failed to fetch hero slides');
      const data = await response.json();
      return data.data as HeroSlide[];
    },
    staleTime: 1000 * 60 * 5, // 5 minutes
  });
}
