'use client';

import { useState } from 'react';

interface UploadImageOptions {
  gender?: string;
  oldImage?: string;
}

interface UploadImageResponse {
  success: boolean;
  message: string;
  data?: {
    url: string;
    filename: string;
  };
}

export function useImageUpload() {
  const [uploading, setUploading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const uploadImage = async (
    file: File,
    options: UploadImageOptions = {}
  ): Promise<string | null> => {
    setUploading(true);
    setError(null);

    try {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('gender', options.gender || 'other');

      if (options.oldImage) {
        formData.append('oldImage', options.oldImage);
      }

      const response = await fetch('/api/admin/upload/teachers', {
        method: 'POST',
        body: formData,
      });

      const data: UploadImageResponse = await response.json();

      if (!response.ok) {
        throw new Error(data.message || 'Failed to upload image');
      }

      return data.data?.url || null;
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Upload failed';
      setError(errorMessage);
      console.error('Upload error:', err);
      throw err;
    } finally {
      setUploading(false);
    }
  };

  return {
    uploading,
    error,
    uploadImage,
  };
}
