'use client';

import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import * as Yup from 'yup';

export interface LeadershipFormData {
  name: string;
  position: string;
  shortDescription?: string; // For home page (1-2 sentences)
  bio?: string; // Full bio for team page
  image?: string;
  expertise?: string;
  order?: number;
  active?: boolean;
}

export interface Leadership extends LeadershipFormData {
  id: string;
  createdAt: Date;
  updatedAt: Date;
}

// Validation schema
export const leadershipSchema = Yup.object().shape({
  name: Yup.string().required('Name is required').min(3).max(100),
  position: Yup.string().required('Position is required').min(2).max(100),
  shortDescription: Yup.string().max(300, 'Short description should be max 300 characters'),
  bio: Yup.string().max(2000),
  image: Yup.string().nullable().test('is-valid-image-url', 'Invalid image URL', (value) => {
    if (!value) return true;
    return value.startsWith('/') || value.startsWith('http://') || value.startsWith('https://');
  }),
  expertise: Yup.string().max(2000),
  order: Yup.number().min(0).max(1000),
  active: Yup.boolean(),
});

// API Response interface
interface ApiResponse<T = Record<string, unknown>> {
  success: boolean;
  message: string;
  data?: T;
  errors?: Record<string, string>;
}

// Fetch all leadership members
export function useLeadership() {
  return useQuery({
    queryKey: ['leadership'],
    queryFn: async () => {
      const response = await fetch('/api/admin/leadership');
      if (!response.ok) throw new Error('Failed to fetch leadership');
      const data: ApiResponse<Leadership[]> = await response.json();
      return data.data || [];
    },
  });
}

// Create leadership member
export function useCreateLeadership() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (values: LeadershipFormData) => {
      const response = await fetch('/api/admin/leadership', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(values),
      });

      const data: ApiResponse<Leadership> = await response.json();
      if (!response.ok) throw new Error(data.message);
      return data.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['leadership'] });
    },
  });
}

// Update leadership member
export function useUpdateLeadership() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ id, values }: { id: string; values: LeadershipFormData }) => {
      const response = await fetch(`/api/admin/leadership/${id}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(values),
      });

      const data: ApiResponse<Leadership> = await response.json();
      if (!response.ok) throw new Error(data.message);
      return data.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['leadership'] });
    },
  });
}

// Delete leadership member
export function useDeleteLeadership() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (id: string) => {
      const response = await fetch(`/api/admin/leadership/${id}`, {
        method: 'DELETE',
      });

      const data: ApiResponse = await response.json();
      if (!response.ok) throw new Error(data.message);
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['leadership'] });
    },
  });
}
