import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";

interface NewsArticle {
  id: string;
  title: string;
  content: string;
  excerpt: string;
  status: "draft" | "published";
  featuredImage?: string;
  author?: string;
  authorId?: string;
  category?: string;
  theme?: string;
  readTime?: number;
  slug: string;
  publishedAt?: string;
  createdAt: string;
  updatedAt: string;
}

// Fetch all articles with pagination and search
export function useNewsArticles(page: number = 1, search: string = "") {
  return useQuery({
    queryKey: ["news", page, search],
    queryFn: async () => {
      const params = new URLSearchParams({
        page: page.toString(),
        limit: "10",
        search,
      });
      const res = await fetch(`/api/admin/news?${params}`);
      if (!res.ok) throw new Error("Failed to fetch articles");
      const result = await res.json();
      return { ...result, articles: result.data };
    },
  });
}

// Fetch single article
export function useNewsArticle(id: string) {
  return useQuery({
    queryKey: ["news", id],
    queryFn: async () => {
      const res = await fetch(`/api/admin/news/${id}`);
      if (!res.ok) throw new Error("Failed to fetch article");
      const { article } = await res.json();
      return article;
    },
    enabled: !!id,
  });
}

// Create article mutation
export function useCreateNewsArticle() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (newArticle: Partial<NewsArticle>) => {
      const res = await fetch("/api/admin/news", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(newArticle),
      });
      if (!res.ok) {
        const error = await res.json();
        throw new Error(error.message || "Failed to create article");
      }
      const { article } = await res.json();
      return article;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });
}

// Update article mutation
export function useUpdateNewsArticle() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async ({ id, data }: { id: string; data: Partial<NewsArticle> }) => {
      const res = await fetch(`/api/admin/news?id=${id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data),
      });
      if (!res.ok) {
        const error = await res.json();
        throw new Error(error.message || "Failed to update article");
      }
      const { article } = await res.json();
      return article;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });
}

// Delete article mutation
export function useDeleteNewsArticle() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (id: string) => {
      const res = await fetch(`/api/admin/news/${id}`, {
        method: "DELETE",
      });
      if (!res.ok) {
        const error = await res.json();
        throw new Error(error.message || "Failed to delete article");
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["news"] });
    },
  });
}
