/**
 * Newsletter Admin Hook
 * React Query hooks for managing newsletter subscribers
 */

import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { useState } from 'react';

// Types
export interface NewsletterSubscriber {
  id: string;
  email: string;
  firstName: string | null;
  subscribed: boolean;
  subscribedAt: Date;
  unsubscribedAt: Date | null;
  createdAt: Date;
  updatedAt: Date;
}

export interface NewsletterStats {
  total: number;
  active: number;
  unsubscribed: number;
}

export interface NewsletterPagination {
  page: number;
  limit: number;
  total: number;
  totalPages: number;
}

export interface NewsletterResponse {
  subscribers: NewsletterSubscriber[];
  stats: NewsletterStats;
  pagination: NewsletterPagination;
}

export interface NewsletterFilters {
  page?: number;
  limit?: number;
  search?: string;
  status?: 'all' | 'subscribed' | 'unsubscribed';
  sortBy?: string;
  sortOrder?: 'asc' | 'desc';
}

// API Functions
async function fetchNewsletterSubscribers(filters: NewsletterFilters): Promise<NewsletterResponse> {
  const params = new URLSearchParams();
  
  if (filters.page) params.append('page', filters.page.toString());
  if (filters.limit) params.append('limit', filters.limit.toString());
  if (filters.search) params.append('search', filters.search);
  if (filters.status) params.append('status', filters.status);
  if (filters.sortBy) params.append('sortBy', filters.sortBy);
  if (filters.sortOrder) params.append('sortOrder', filters.sortOrder);

  const response = await fetch(`/api/admin/newsletter?${params.toString()}`);
  const data = await response.json();
  
  if (!response.ok) {
    throw new Error(data.message || 'Failed to fetch newsletter subscribers');
  }
  
  return data.data;
}

async function updateSubscriberStatus(id: string, subscribed: boolean): Promise<NewsletterSubscriber> {
  const response = await fetch(`/api/admin/newsletter/${id}`, {
    method: 'PATCH',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ subscribed }),
  });
  
  const data = await response.json();
  
  if (!response.ok) {
    throw new Error(data.message || 'Failed to update subscriber status');
  }
  
  return data.data;
}

async function deleteSubscriber(id: string): Promise<void> {
  const response = await fetch(`/api/admin/newsletter/${id}`, {
    method: 'DELETE',
  });
  
  const data = await response.json();
  
  if (!response.ok) {
    throw new Error(data.message || 'Failed to delete subscriber');
  }
}

async function bulkDeleteSubscribers(ids: string[]): Promise<{ deletedCount: number }> {
  const response = await fetch('/api/admin/newsletter', {
    method: 'DELETE',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ ids }),
  });
  
  const data = await response.json();
  
  if (!response.ok) {
    throw new Error(data.message || 'Failed to delete subscribers');
  }
  
  return data.data;
}

async function exportSubscribers(status: 'all' | 'subscribed' | 'unsubscribed' = 'all', ids: string[] = []): Promise<Blob> {
  const response = await fetch('/api/admin/newsletter/export', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ status, ids }),
  });
  
  if (!response.ok) {
    const data = await response.json();
    throw new Error(data.message || 'Failed to export subscribers');
  }
  
  return response.blob();
}

// React Query Hooks

/**
 * Hook to fetch newsletter subscribers with filters
 */
export function useNewsletterSubscribers(initialFilters: NewsletterFilters = {}) {
  const [filters, setFilters] = useState<NewsletterFilters>({
    page: 1,
    limit: 10,
    search: '',
    status: 'all',
    sortBy: 'createdAt',
    sortOrder: 'desc',
    ...initialFilters,
  });

  const query = useQuery({
    queryKey: ['newsletter-subscribers', filters],
    queryFn: () => fetchNewsletterSubscribers(filters),
    staleTime: 1000 * 60 * 1, // 1 minute
  });

  return {
    ...query,
    filters,
    setFilters,
    // Helper functions
    setPage: (page: number) => setFilters((prev) => ({ ...prev, page })),
    setSearch: (search: string) => setFilters((prev) => ({ ...prev, search, page: 1 })),
    setStatus: (status: 'all' | 'subscribed' | 'unsubscribed') => setFilters((prev) => ({ ...prev, status, page: 1 })),
    setSort: (sortBy: string, sortOrder: 'asc' | 'desc') => setFilters((prev) => ({ ...prev, sortBy, sortOrder })),
  };
}

/**
 * Hook to update subscriber status (subscribe/unsubscribe)
 */
export function useUpdateSubscriberStatus() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: ({ id, subscribed }: { id: string; subscribed: boolean }) =>
      updateSubscriberStatus(id, subscribed),
    onSuccess: () => {
      // Invalidate and refetch
      queryClient.invalidateQueries({ queryKey: ['newsletter-subscribers'] });
    },
  });
}

/**
 * Hook to delete a single subscriber
 */
export function useDeleteSubscriber() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (id: string) => deleteSubscriber(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['newsletter-subscribers'] });
    },
  });
}

/**
 * Hook to bulk delete subscribers
 */
export function useBulkDeleteSubscribers() {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: (ids: string[]) => bulkDeleteSubscribers(ids),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['newsletter-subscribers'] });
    },
  });
}

/**
 * Hook to export subscribers to CSV
 */
export function useExportSubscribers() {
  return useMutation({
    mutationFn: ({ status, ids }: { status?: 'all' | 'subscribed' | 'unsubscribed'; ids?: string[] }) =>
      exportSubscribers(status, ids),
    onSuccess: (blob) => {
      // Trigger download
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `newsletter-subscribers-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    },
  });
}
