import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';

export interface SiteSettings {
  logo_url?: string;
  logo_text?: string;
  logo_initials?: string;
  favicon_url?: string;
  logo_alt?: string;
  show_logo_text?: string; // 'true' or 'false' as string
  [key: string]: string | undefined;
}

// Fetch site settings (admin)
export function useSiteSettings(keys?: string[]) {
  return useQuery({
    queryKey: ['site-settings', keys],
    queryFn: async () => {
      const url = keys
        ? `/api/admin/site-settings?keys=${keys.join(',')}`
        : '/api/admin/site-settings';
      
      const response = await fetch(url);
      if (!response.ok) throw new Error('Failed to fetch site settings');
      const data = await response.json();
      return data.data as SiteSettings;
    },
  });
}

// Fetch public site settings
export function usePublicSiteSettings(keys?: string[]) {
  return useQuery({
    queryKey: ['public-site-settings', keys],
    queryFn: async () => {
      const url = keys
        ? `/api/site-settings?keys=${keys.join(',')}`
        : '/api/site-settings';
      
      const response = await fetch(url);
      if (!response.ok) throw new Error('Failed to fetch site settings');
      const data = await response.json();
      return data.data as SiteSettings;
    },
    staleTime: 1000 * 60 * 10, // 10 minutes cache for public settings
  });
}

// Update site settings
export function useUpdateSiteSettings() {
  const queryClient = useQueryClient();
  
  return useMutation({
    mutationFn: async (settings: Record<string, string>) => {
      const response = await fetch('/api/admin/site-settings', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ settings }),
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || 'Failed to update site settings');
      }
      
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['site-settings'] });
      queryClient.invalidateQueries({ queryKey: ['public-site-settings'] });
    },
  });
}
