import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { StudentInNeed } from '@/types/types';

// Fetch all students in need
export const useStudentInNeed = () => {
  return useQuery({
    queryKey: ['students-in-need'],
    queryFn: async () => {
      const response = await fetch('/api/admin/students/in-need');
      if (!response.ok) throw new Error('Failed to fetch students in need');
      const data = await response.json();
      return data.data as StudentInNeed[];
    },
  });
};

// Fetch single student in need
export const useStudentInNeedDetail = (id: string) => {
  return useQuery({
    queryKey: ['student-in-need', id],
    queryFn: async () => {
      const response = await fetch(`/api/admin/students/in-need/${id}`);
      if (!response.ok) throw new Error('Failed to fetch student in need');
      const data = await response.json();
      return data.data as StudentInNeed;
    },
    enabled: !!id,
  });
};

// Create student in need
export const useCreateStudentInNeed = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (student: Omit<StudentInNeed, 'id' | 'createdAt' | 'updatedAt'>) => {
      const response = await fetch('/api/admin/students/in-need', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(student),
      });

      const data = await response.json();
      if (!response.ok) throw new Error(data.message || 'Failed to create student in need');
      return data.data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['students-in-need'] });
    },
  });
};

// Update student in need
export const useUpdateStudentInNeed = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async ({ id, student }: { id: string; student: Omit<StudentInNeed, 'id' | 'createdAt' | 'updatedAt'> }) => {
      const response = await fetch(`/api/admin/students/in-need/${id}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(student),
      });

      const data = await response.json();
      if (!response.ok) throw new Error(data.message || 'Failed to update student in need');
      return data.data;
    },
    onSuccess: (_, { id }) => {
      queryClient.invalidateQueries({ queryKey: ['students-in-need'] });
      queryClient.invalidateQueries({ queryKey: ['student-in-need', id] });
    },
  });
};

// Delete student in need
export const useDeleteStudentInNeed = () => {
  const queryClient = useQueryClient();

  return useMutation({
    mutationFn: async (id: string) => {
      const response = await fetch(`/api/admin/students/in-need/${id}`, {
        method: 'DELETE',
      });

      const data = await response.json();
      if (!response.ok) throw new Error(data.message || 'Failed to delete student in need');
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['students-in-need'] });
    },
  });
};
